<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\TSeo;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="navigations__navigation_text", indexes={@ORM\Index(name="navigation", columns={"lang", "navigation_id", "is_published"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"NavigationTextListener"})
 */
class NavigationText
{
	use TSeo;

	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Navigation", inversedBy="texts")
	 * @ORM\JoinColumn(name="navigation_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Navigation $navigation;

	/**
	 * @ORM\Id
	 * @ORM\Column(type="string", length=2, nullable=true)
	 */
	protected ?string $lang = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=false)
	 */
	public string $title;

	/**
	 * @ORM\Column(type="string", length=255, nullable=false)
	 */
	protected string $alias;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default":1})
	 */
	public int $isPublished;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default":0})
	 */
	public int $isHomepage;

	/**
	 * @ORM\Column(type="string", length=400, nullable=true)
	 */
	public ?string $img = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $description = null;

	/**
	 * @ORM\Column(type="array", nullable=true)
	 */
	public ?array $params = null;

	/**
	 * @ORM\Column(name="custom_text1", type="string", length=255, nullable=true)
	 */
	public ?string $customText1 = null;

	/**
	 * @ORM\Column(name="custom_text2", type="string", length=255, nullable=true)
	 */
	public ?string $customText2 = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $fullUrl = null;

	public function __construct(Navigation $navigation, string $title, ?string $img = null, ?string $description, string $lang)
	{
		$this->navigation  = $navigation;
		$this->lang        = $lang;
		$this->title       = $title;
		$this->isPublished = 1;
		$this->isHomepage  = 0;
		$this->img         = $img;
		$this->description = $description;
		$this->params      = [];
		$this->setAlias($title);
		$this->setSeo([]);
	}

	public function getLang(): string
	{
		return $this->lang;
	}

	public function setLang(?string $lang): self
	{
		$this->lang = $lang === '' ? null : $lang;

		return $this;
	}

	public function getTitle(): string
	{
		return $this->title;
	}

	public function setTitle(string $title): self
	{
		$this->title = $title;

		if (!$this->alias) {
			$this->setAlias($title);
		}

		return $this;
	}

	public function getAlias(): ?string
	{
		return $this->alias;
	}

	public function setAlias(string $alias): self
	{
		if ($alias !== '') {
			$this->alias = Strings::webalize($alias);
		} else {
			$this->alias = Strings::webalize($this->title);
		}

		return $this;
	}

	public function getImg(): ?string
	{
		return $this->img;
	}

	public function setImg(?string $img): self
	{
		$this->img = $img;

		return $this;
	}

	public function getDescription(): ?string
	{
		return $this->description;
	}

	public function setDescription(?string $description): self
	{
		$this->description = $description;

		return $this;
	}

	public function getParams(): array
	{
		return $this->params ?? [];
	}

	/**
	 * @param int|string $key
	 * @return mixed|null
	 */
	public function getParam($key)
	{
		return $this->params[$key] ?? null;
	}

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}

}
