<?php declare(strict_types = 1);

namespace Navigations\Model\Helper;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use Contributte\Translation\Translator;
use Doctrine\ORM\EntityRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Request;
use Nette\Utils\Strings;
use Throwable;

class NavigationsHelper
{
	protected EntityManagerDecorator $em;
	protected Storage                $cacheStorage;
	protected ?Cache                 $cache = null;
	protected Translator             $translator;
	protected Sites                  $sitesService;
	protected Request                $httpRequest;

	protected array $cPublished = [];

	protected array $cacheDep = [
		Cache::TAGS   => [Navigations::CACHE_NAMESPACE],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(EntityManagerDecorator $em, Storage $IStorage, Translator $translator, Sites $sites,
	                            Request                $request)
	{
		$this->em           = $em;
		$this->cacheStorage = $IStorage;
		$this->translator   = $translator;
		$this->sitesService = $sites;
		$this->httpRequest  = $request;
	}

	protected function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	protected function getEr(): EntityRepository
	{
		return $this->em->getRepository(Navigation::class);
	}

	public function getSitesService(): Sites { return $this->sitesService; }

	/**
	 * @return DaoNavigationItem[]
	 * @throws Throwable
	 */
	public function getPublished(?string $lang = null): array
	{
		$lang      = $lang ?: $this->translator->getLocale();
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();

		$key = $siteIdent . '-' . $lang;

		if (!isset($this->cPublished[$key])) {
			$this->cPublished[$key] = $this->getCache()->load('published_' . $key, function(&$dep) use ($siteIdent, $lang) {
				$dep  = $this->cacheDep;
				$data = [];

				foreach ($this->getEr()->createQueryBuilder('n')
					         ->select('n, nt, g, IDENTITY(n.parent) as parent')
					         ->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang AND nt.isPublished = 1')
					         ->join('n.group', 'g')
					         ->andWhere('n.lvl > 0')
					         ->andWhere('n.site = :site')
					         ->setParameters([
						         'lang' => $lang,
						         'site' => $siteIdent,
					         ])
					         ->orderBy('n.parent')->addOrderBy('n.lft')
					         ->getQuery()->getArrayResult() as $k => $v) {
					$arr              = $v[0];
					$arr['texts']     = $arr['texts'][$lang];
					$arr['parent']    = $v['parent'];
					$arr['groupType'] = $arr['group']['type'];

					$data[] = $this->fillDao($arr);
				}

				return $data;
			});
		}

		return $this->cPublished[$key];
	}

	public function fillDao(array $item, ?DaoNavigationItem $parent = null): DaoNavigationItem
	{
		$dao                  = new DaoNavigationItem();
		$dao->id              = (int) $item['id'];
		$dao->title           = $item['texts']['title'];
		$dao->alias           = $item['texts']['alias'];
		$dao->isHomepage      = (int) $item['texts']['isHomepage'];
		$dao->groupType       = $item['group']['type'];
		$dao->componentType   = $item['componentType'];
		$dao->componentParams = (array) $item['componentParams'];
		$dao->ico             = $item['icon'];
		$dao->icoSecondary    = $item['iconSecondary'];
		$dao->lang            = $item['texts']['lang'];
		$dao->description     = $item['texts']['description'];
		$dao->img             = $item['texts']['img'];
		$dao->seo             = $item['texts']['seo'];
		$dao->customText1     = $item['texts']['customText1'];
		$dao->customText2     = $item['texts']['customText2'];
		$dao->parentId        = (int) $item['parent'];
		$dao->isParent        = $dao->parentId ? true : false;
		$dao->class           = $item['params']['linkClass'] ?? '';
		$dao->lvl             = (int) $item['lvl'];
		$dao->setModified($item['modified'] ?: $item['created']);
		$dao->setParams($item['params'] ?: []);

		if (isset($item['params']['openIn'])) {
			$dao->openIn = $item['params']['openIn'];
			unset($item['params']['openIn']);
		}

		if ($item['texts']['fullUrl']) {
			$dao->link             = '/' . $item['texts']['fullUrl'];
			$dao->isFullLinkFilled = true;
		}

		return $dao;
	}

	/**
	 * @return DaoNavigationItem[]
	 * @throws Throwable
	 */
	public function getNavs(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		return $this->getCache()->load('publishedNavs/' . $lang) ?: $this->getPublished($lang);
	}

	public function getHttpRequest(): Request { return $this->httpRequest; }

	public static function removeUrlSuffix(string $url): string
	{
		$endSuffix = NavigationConfig::load('urlSuffix');
		if ($endSuffix && Strings::endsWith($url, $endSuffix)) {
			return Strings::substring($url, 0, -strlen($endSuffix));
		}

		return $url;
	}
}
