<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\Exceptions\EntityContainChildren;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Navigations\AdminModule\Model\NavigationsGroups;
use Navigations\Model\Entities\NavigationGroup;
use Nette\Utils\Html;

class NavigationsGroupsGrid extends BaseControl
{
	/** @var INavigationGroupFormFactory */
	protected $formFactory;

	/** @var NavigationsGroups */
	protected $navigationsGroupsService;

	public function __construct(INavigationGroupFormFactory $formFactory, NavigationsGroups $navigationsGroups)
	{
		$this->formFactory              = $formFactory;
		$this->navigationsGroupsService = $navigationsGroups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile() ?: __DIR__ . '/NavigationGroupForm.latte');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();

		try {
			if ($this->navigationsGroupsService->remove($id)) {
				$presenter->flashMessageSuccess('default.removed');
			} else {
				$presenter->flashMessageDanger('default.removeFailed');
			}
		} catch (EntityContainChildren $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleEdit(int $id)
	{
		$presenter = $this->getPresenter();
		$group     = $this->navigationsGroupsService->get($id);

		if (!$group)
			$presenter->error();

		$this['groupForm']->groupId       = $id;
		$this->template->modalTitle       = $this->t('navigations.title.editNavigationGroup',
			['title' => $group->getText() ? $group->getText()->title : '']);
		$this->template->modal            = 'groupForm';
		$this->template->modalDialogClass = 'modal-lg';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->navigationsGroupsService->getEr()->createQueryBuilder('g')
			->addSelect('gt')
			->leftJoin('g.texts', 'gt', 'WITH', 'gt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('gt.title');

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('title', 'navigations.navigation.title', 'text.title')
			->setRenderer(function(NavigationGroup $row) {
				return Html::el('a', [
					'href'  => $this->link('edit!', $row->getId()),
					'class' => 'ajax',
				])->setText($row->getText()->title);
			});

		// Filter
		$grid->addFilterText('title', 'navigations.navigation.title', 'gt.title');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentGroupForm(): NavigationGroupForm
	{
		$control = $this->formFactory->create();
		if ($this->getParameter('id'))
			$control->setNavigationGroup((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
