<?php declare(strict_types = 1);

namespace Navigations\Model;

use Contributte\EventDispatcher\EventDispatcher;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseEntityService;
use Core\FrontModule\Model\Redirects;
use Core\Model\Lang\DefaultLang;
use Core\Model\Lang\Langs;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationGroupText;
use Navigations\Model\Entities\NavigationText;
use Navigations\Model\Event\RouteInEvent;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;

/**
 * Class Navigations
 * @package Navigations\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	/** @var string */
	const CACHE_NAMESPACE = 'navigations';

	/** @var string */
	protected $entityClass = Navigation::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var DaoNavigationItem[][] */
	public $cNavs;

	/** @var DaoNavigationItem[][] */
	public $cNavsTree;

	/** @var ITranslator @inject */
	public $translator;

	/** @var ItemsCollector @inject */
	public $itemsCollector;

	/** @var Redirects @inject */
	public $redirectsService;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Langs @inject */
	public $langsService;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var NavigationsHelper @inject */
	public $navigationsHelper;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	protected function getCurrentSiteIdent(): string { return $this->navigationsHelper->getSitesService()->getCurrentSite()->getIdent(); }

	/**
	 * @param int    $id
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPath(int $id, ?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();
		$key  = implode('/', ['path', $this->getCurrentSiteIdent(), $id, $lang]);

		return $this->getCache()->load($key, function(&$dep) use ($id, $lang) {
			$dep = $this->cacheDep;

			$result = [];
			$stmt   = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM navigations__navigation WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, navigations__navigation h WHERE @r <> 0) T1 
						JOIN navigations__navigation T2 ON T1._id = T2.id ORDER BY T1.lvl DESC");
			$stmt->execute();

			foreach ($stmt->fetchAll() as $row) {
				$nav = $this->getNavigation((int) $row['id'], $lang);
				if ($nav)
					$result[$row['id']] = $nav;
			}

			return $result;
		});
	}

	/**
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedToDaoNavigationItem(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavs[$lang])) {
			$key = 'publishedNavs/' . $this->getCurrentSiteIdent() . '/' . $lang;

			$this->cNavs[$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep    = $this->cacheDep;
				$result = [];

				$loop = function(array $arr) use (&$loop, &$result) {
					/** @var DaoNavigationItem[] $arr */
					foreach ($arr as $v) {
						$result[$v->id] = $v;

						if ($v->childs)
							$loop($v->childs);
					}
				};

				foreach ($this->getPublishedDaoNavigationStructure($lang) as $v)
					$loop($v);

				return $result;
			});
		}

		return $this->cNavs[$lang];
	}

	public function getPublishedDaoNavigationStructure(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavsTree[$lang])) {
			$key = 'publishedStructure/' . $this->getCurrentSiteIdent() . $lang;

			$this->cNavsTree[$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep = $this->cacheDep;
				/** @var DaoNavigationItem[] $flat */
				$flat    = [];
				$grouped = [];
				$tree    = [];

				foreach ($this->navigationsHelper->getPublished($lang) as $nav) {
					$grouped[$nav->groupType][$nav->parentId][$nav->id] = $nav;
					$flat[$nav->id]                                     = &$grouped[$nav->groupType][$nav->parentId][$nav->id];
				}
				$this->em->clear([
					Navigation::class,
					NavigationText::class,
					NavigationGroup::class,
					NavigationGroupText::class,
				]);

				$createTree = function(&$list, $parent) use (&$createTree) {
					$tree = [];
					foreach ($parent as $k => $l) {
						if (isset($list[$l->id])) {
							if ($l->lvl >= 1)
								foreach ($list[$l->id] as $kk => $v) {
									$list[$l->id][$kk]->parent = &$parent[$k];
								}
							$l->isParent = true;
							$l->childs   = $createTree($list, $list[$l->id]);
						}

						$tree[] = $l;
					}

					return $tree;
				};

				foreach ($this->getEr()->createQueryBuilder('n')->select('n.id, ng.type')
					         ->join('n.group', 'ng')
					         ->where('n.lvl = 0')
					         ->andWhere('n.site = :site')
					         ->setParameters([
						         'site' => $this->navigationsHelper->getSitesService()->getCurrentSite()->getIdent(),
					         ])->getQuery()->getScalarResult() as $row) {
					if (!isset($grouped[$row['type']]))
						continue;

					$dao                        = new DaoNavigationItem();
					$dao->id                    = $row['id'];
					$dao->parent                = 0;
					$grouped[$row['type']][0][] = $dao;

					$tree[$row['type']] = $createTree($grouped[$row['type']], $grouped[$row['type']][0])[0]->childs;
				}

				$aliases = [];

				foreach ($flat as $k => $v) {
					if ($v->componentType == 'navigation.alias') {
						$flat[$k]->setParam('aliasReference', $flat[$v->componentParams['navigation']]);
						$flat[$k]->link = $flat[$v->componentParams['navigation']]->link;
						$aliases[$k]    = $v;
						continue;
					}

					$flat[$k]->link = $this->generateLink($v);
				}

				foreach ($aliases as $k => $v) {
					$flat[$k]->link = $this->generateLink($v);
				}

				return $tree;
			});
		}

		return $this->cNavsTree[$lang];
	}

	/**
	 * @param string $component
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedByComponent(string $component, ?string $lang = null): array
	{
		$result = [];
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $nav) {
			if ($nav->componentType === $component)
				$result[$nav->id] = $nav;
		}

		return $result;
	}

	public function getNavigation(int $id, ?string $lang = null): ?DaoNavigationItem
	{
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $nav) {
			if ($nav->id === $id)
				return $nav;
		}

		return null;
	}

	public function findByAlias(array $alias, ?string $lang = null): ?DaoNavigationItem
	{
		$oAlias = $alias;
		$lang   = $lang ?: $this->translator->getLocale();

		$navigation = null;
		$a          = end($alias);

		foreach ($this->getEr()->createQueryBuilder('n')
			         ->select('n.id')
			         ->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang AND nt.isPublished = 1 AND nt.alias = :alias')
			         ->andWhere('n.lvl > 0')
			         ->andWhere('n.componentType != :typeAlias')
			         ->setParameters([
				         'typeAlias' => 'navigation.alias',
				         'lang'      => $lang,
				         'alias'     => $a,
			         ])->getQuery()->getScalarResult() as $row) {
			$path  = array_values($this->getPath((int) $row['id'], $lang));
			$match = true;

			// Vyhledání správné cesty podle nadřazených
			foreach ($alias as $k => $v) {
				if ($v != $path[$k]->alias) {
					$match = false;
					break;
				}
			}

			$n = $this->getNavigation((int) $row['id'], $lang);
			if ($match) {
				$navigation = $n;
				break;
			}

			// Vyhledání správné cesty podle nadřezených 2
			foreach ($path as $k => $p) {
				if ($p->alias != $alias[$k]) {
					$match = false;
					break;
				}
			}

			if ($match) {
				$navigation = $n;
				break;
			}
		}

		// Vyhledání v přesměrování
		if (!$navigation) {
			$redirect = $this->redirectsService->getEr()->findOneBy(['from'        => implode('/', $alias),
			                                                         'package'     => 'Navigations',
			                                                         'relationKey' => 'Navigation']);

			if ($redirect && $redirect->relationValue !== null)
				$navigation = $this->getNavigation((int) $redirect->relationValue);
			else {
				array_pop($oAlias);

				if ($oAlias) {
					$navigation = $this->findByAlias($oAlias, $lang);
				}
			}
		}

		return $navigation;
	}

	public function generateLink(DaoNavigationItem $navigation): ?string
	{
		$t = [
			'locale'           => $navigation->lang,
			'activeNavigation' => $navigation,
		];

		return $this->getUrlById($t);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function getHomepage(?string $lang = null): ?DaoNavigationItem
	{
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $v) {
			if ($v->isHomepage)
				return $v;
		}

		return null;
	}

	public function getHomepageForRouter(?string $lang = null): array
	{
		$hp = $this->getHomepage($lang);
		if ($hp) {
			$component = $this->itemsCollector->getItemById($hp->componentType);

			return $component->routerIn($hp, ['getHomepage' => 1]) + ['activeNavigation' => $hp];
		}

		return [];
	}

	public function getIdByUrl($url)
	{
		try {
			$return = [];
			$locale = $url['locale'] ?? $this->defaultLang->locale;
			$key    = self::CACHE_NAMESPACE . '/idByUrl/' . $url['path'] . '/' . $this->getCurrentSiteIdent() . '/' . $locale;

			return $this->getCache()->load($key, function(&$dep) use ($url, $return, $locale) {
				$dep = $this->cacheDep;

				$navigation = null;

				if (isset($url['path'])) {
					$tmp = explode('/', $url['path']);
					$t   = $tmp;

					// Upravení pro stránkování u obecné komponenty
					$paginatorIndex = array_search($this->translator->translate('default.urlPart.page'), $t);
					if ($paginatorIndex !== false) {
						$return['list-page'] = $t[$paginatorIndex + 1] ?? 1;
						$return['do']        = 'list-paginator';
						unset($t[$paginatorIndex + 1]);
						unset($t[$paginatorIndex]);
						unset($tmp[$paginatorIndex + 1]);
						unset($tmp[$paginatorIndex]);
					}

					if (!$navigation) {
						$navigation = $this->findByAlias($t, $locale);
					}

					$url['path'] = implode('/', $tmp);
				}

				if ($navigation && $component = $this->itemsCollector->getItemById($navigation->componentType)) {
					$out = $component->routerIn($navigation, $url);

					if (is_array($out))
						$return += $out + ['activeNavigation' => $navigation];
				} else if (!$navigation) {
					$this->eventDispatcher->dispatch(Navigations::class . '::routeInNotFound', new RouteInEvent(null, $url, $return));

					if (!empty($return))
						return $return;

					$hp        = $this->getHomepage($url['locale']);
					$component = $this->itemsCollector->getItemById($hp->componentType);

					if (!$component)
						return null;

					$tmp = $component->routerIn($hp, $url);

					if (!$tmp)
						return null;

					$return += $tmp;
					$return += ['activeNavigation' => $hp];
				}

				$component->updateCacheDep($dep, $return);

				return $return;
			});
		} catch (\Exception $e) {
			return null;
		}
	}

	public function getUrlById(&$params)
	{
		$return = null;
		try {
			$tmp = $params;
			if ($tmp['activeNavigation']) {
				$tmp['activeNavigation'] = $tmp['activeNavigation']->id;
			}

			foreach ($tmp as $v)
				if (is_object($v))
					unset($tmp[$v]);

			$key = 'urlById/' . $this->getCurrentSiteIdent() . '/' . serialize($tmp);

			$data = $this->getCache()->load($key, function(&$dep) use ($params, $return) {
				$dep = $this->cacheDep;

				if (isset($params['activeNavigation'])) {
					$navType = $params['activeNavigation']->componentType;

					if ($navType == 'navigation.alias') {
						$tmp = $params['activeNavigation']->getParam('aliasReference');

						if ($tmp) {
							$params['activeNavigation'] = $tmp;
							$navType                    = $tmp->componentType;
						}
					}
					$component = $this->itemsCollector->getItemById($navType);

					if ($component)
						$return = $this->getPreAlias($params['activeNavigation']) . $component->routerOut($params['activeNavigation'], $params);
				}

				if (!$return) {
					$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);

					if (!$groups) {
						if ($params['presenter']) {
							preg_match('/Override$/', $params['presenter'], $matches, PREG_OFFSET_CAPTURE, 0);

							if (isset($matches[0]))
								$params['presenter'] = substr($params['presenter'], 0, $matches[0][1]);
						}

						$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);
					}

					if (!$groups)
						return null;

					ksort($groups);

					foreach ($groups as $components) {
						/** @var INavigationItem[]|BaseItem[] $components */
						foreach ($components as $component) {
							foreach ($this->getPublishedByComponent($component->getId(), $params['locale']) as $navigation) {
								$return = $component->routerOut($navigation, $params);

								if ($return === '')
									$return = null;
								else if ($return !== null && $return !== false) {
									$component->updateCacheDep($dep, $params);
									$return = $this->getPreAlias($navigation) . $return;
									break 3;
								}
							}
						}
					}
				}

				// Upravení url pro stránkování
				if (isset($params['do']) && isset($params['list-page']) && $params['do'] == 'list-paginator') {
					if ($params['list-page'] > 1) {
						$return .= '/' . $this->translator->translate('default.urlPart.page') . '/' . $params['list-page'];
					}

					unset($params['list-page']);
					unset($params['do']);
				}

				if (isset($params['activeNavigation']) && !$params['activeNavigation']->link)
					$params['activeNavigation']->link = $return;

				return ['return' => $return, 'params' => $params];
			});

			$params = $data['params'];
			$return = $data['return'];

			return $return;
		} catch (\Exception $e) {
		}

		return null;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 *
	 * @return string
	 */
	public function getPreAlias(DaoNavigationItem $navigation)
	{
		$arr = [];
		$t   = $navigation->parent;

		while ($t) {
			$arr[] = $t->alias;
			$t     = $t->parent;
		}

		$url = $arr ? '/' . implode('/', array_reverse($arr)) : '';
		$this->setLocaleUrlPrefix($url, $navigation);

		return $url;
	}

	public function setLocaleUrlPrefix(string &$url, ?DaoNavigationItem $navigation = null): void
	{
		if ($navigation->getLang() && $this->translator->getDefaultLocale() != $navigation->getLang())
			$url = '/' . $navigation->getLang() . $url;
	}

	/*******************************************************************************************************************
	 * ===========================  Cache
	 */
}
