<?php declare(strict_types = 1);

namespace Pages\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Nette\Caching\Cache;
use Pages\FrontModule\Model\Dao\DaoTemplatePage;
use Pages\Model\Entities\TemplatePage;
use Pages\Model\Paths;

/**
 * Class TemplatePages
 * @package Pages\FrontModule\Model
 */
class TemplatePages extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'templatePages';

	protected $entityClass = TemplatePage::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var Paths */
	protected $paths;

	public function __construct(Paths $paths)
	{
		$this->paths = $paths;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function get(int $id): ?DaoTemplatePage
	{
		$lang = $this->translator->getLocale();
		$key  = 'templatePage/' . $id . '/' . $lang;

		return $this->getCache()->load($key, function(&$dep) use ($id, $lang) {
			$dep = $this->cacheDep;

			$templatePage = $this->getEr()->createQueryBuilder('tp')
				->select('tp.id, tp.title, tpt.texts, tp.template')
				->join('tp.texts', 'tpt', 'WITH', 'tpt.lang = :lang')
				->where('tp.id = :id')
				->setParameters([
					'id'   => $id,
					'lang' => $lang,
				])
				->getQuery()->getOneOrNullResult();

			return $templatePage ? $this->fillDao($templatePage) : null;
		});
	}

	public function getTemplatePathByFile(string $file): string
	{
		return $this->paths->getTemplatePagesFile($file);
	}

	public function getTemplatePathById(int $id): ?string
	{
		$template = $this->get($id);

		return $template ? $this->paths->getTemplatePagesFile($template->template) : null;
	}

	protected function fillDao(array $data): DaoTemplatePage
	{
		$dao = new DaoTemplatePage((int) $data['id'], $data['title'], $data['texts'], $data['template']);

		return $dao;
	}
}
