<?php declare(strict_types = 1);

namespace Pages\FrontModule\Model\Search;

use Core\FrontModule\Model\Dao\SearchItemResult;
use Core\FrontModule\Model\Providers\ISearchItem;
use Core\FrontModule\Model\Search\BaseSearchItem;
use Core\Model\Event\TtSearchEvent;
use Core\Model\TemplateReader\TemplateReader;
use Navigations\Model\Entities\Navigation;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Entities\TemplatePage;

class TemplatePageSearchItem extends BaseSearchItem implements ISearchItem
{
	const GROUP = 'templatePage';

	/** @var ITranslator @inject */
	public $translator;

	/** @var TemplatePages @inject */
	public $templatePagesService;

	/** @var TemplateReader @inject */
	public $templateReader;

	public function search(string $q): array
	{
		/** @var SearchItemResult[] $result */
		$result         = [];
		$qOpt           = Strings::webalize($q);
		$explo          = explode(' ', $q);
		$exploOpt       = [];
		$navigationsAll = [];
		$templatePages  = [];

		foreach ($explo as $v)
			$exploOpt[] = Strings::webalize($v);

		// Vyhledání všech templatePage položek v menu
		foreach ($this->em->getRepository(Navigation::class)->createQueryBuilder('n')
			         ->select('n.id, nt.title, n.componentParams')
			         ->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang AND nt.isPublished = 1')
			         ->where('n.componentType = :type')
			         ->setParameters([
				         'type' => 'pages.navigation.templatePage',
				         'lang' => $this->translator->getLocale(),
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$navigationsAll[] = $row;
		}

		// Porovnání titulků
		foreach ($navigationsAll as $row) {
			$templatePageId                 = $row['componentParams']['templatePage'];
			$templatePages[$templatePageId] = $row['id'];

			if (Strings::contains(Strings::webalize($row->title), $qOpt)) {
				$this->addNavToResult($result, $row['id']);
			}
		}

		// Vyhledání v detailu templatePage
		foreach ($this->em->getRepository(TemplatePage::class)->createQueryBuilder('tp')
			         ->select('tp.id, tp.title, tpt.texts')
			         ->join('tp.texts', 'tpt', 'WITH', 'tpt.lang = :lang')
			         ->where('tpt.texts LIKE :text OR tp.title LIKE :text')
			         ->andWhere('tp.id IN (:ids)')
			         ->setParameters([
				         'text' => "%$q%",
				         'ids'  => array_keys($templatePages),
				         'lang' => $this->translator->getLocale(),
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$navId = $templatePages[$row['id']];
			if (!isset($result[$navId])) {
				$this->addNavToResult($result, $navId, $row['title']);
			}
			$result[$navId]->params = ['texts' => $row['texts']];
		}

		// Vyhledání v tt
		$templates = $this->templatePagesService->getEr()->createQueryBuilder('tp')
			->select('tp.id, tp.title, tpt.texts, tp.template')
			->join('tp.texts', 'tpt', 'WITH', 'tpt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->getQuery()->getArrayResult();
		foreach ($navigationsAll as $nav) {
			$template = $templates[$nav['componentParams']['templatePage']] ?? null;

			if (!$template)
				continue;

			$templateFile = $this->templatePagesService->getTemplatePathByFile($template['template']);
			$inputs       = $this->templateReader->getTemplateInputs($templateFile);
			$event        = new TtSearchEvent($q, $inputs, $template['texts']);
			$this->eventDispatcher->dispatch('core.searching.deep', $event);

			foreach ($event->getResult() as $type => $rows) {
				$navId = $nav['id'];
				if (!isset($result[$navId])) {
					$this->addNavToResult($result, $navId, $template['title']);
					if (!isset($result[$navId]->params['tt']))
						$result[$navId]->params['tt'] = [];
				}

				$result[$navId]->params['tt'] = array_merge($result[$navId]->params['tt'], $event->getResult());
			}
		}

		return $result;
	}

	protected function addNavToResult(&$result, $navId, ?string $title = null): void
	{
		$nav = $this->navigations->getNavigation((int) $navId);
		if ($nav) {
			$result[$navId]             = new SearchItemResult(self::GROUP, $title ?: $nav->title, (string) $nav->link);
			$result[$navId]->navigation = $nav;
		}
	}
}
