<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Pages\Model\Entities\TemplateBlock;
use Pages\Model\Entities\TemplateBlockText;
use Pages\Model\Paths;
use Pages\AdminModule\Model\TemplateBlocks;

class TemplateBlockForm extends TemplateReaderControl
{
	/** @var TemplateBlock */
	protected $entity;

	/** @var TemplateBlocks */
	protected $templateBlocksService;

	/** @var Paths */
	protected $pathsService;

	public function __construct(TemplateBlocks $templateBlocks, Paths $paths)
	{
		$this->templateBlocksService = $templateBlocks;
		$this->pathsService          = $paths;
	}

	public function render()
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->pathsService->getTemplateBlocksDir());
		$this->templateReader->setTranslateKey('templateBlock');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addSelect('template', 'default.templates', $templates)->setPrompt('default.choose')->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplateBlockText[] $texts */
			$texts           = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->entity) {
				$entity = $this->entity;
				$texts  = $entity->getTexts()->toArray();
			} else {
				$entity = new TemplateBlock($values->title, $values->template);
			}

			if (!$langValues) {
				if ($this->getPresenter()->getAction() === 'add') {
					$tmp = $form->getHttpData()['component'];
					if (is_array($tmp)) {
						foreach ($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)) as $l => $v)
							$langValues[$l]['component'] = $v;
					}
				} else {
					foreach ($this->langsService->getLangs(false) as $l => $v) {
						$langValues[$l]['component'] = $componentValues;
					}
				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplateBlockText($entity, $l);
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					$v[$name] = $componentValues[$name];
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$entity->setTexts($texts);

			$entity->title    = $values->title;
			$entity->template = $values->template;

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity)->flush();
			$form->addCustomData('templateBlockId', $entity->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setBlock($id)
	{
		$this->entity = $this->templateBlocksService->get($id);

		if (!$this->entity)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form = $this['form'];
		$form->setDefaults([
			'title' => $this->entity->title,
		]);

		if ($this->entity->template && array_key_exists($this->entity->template, $form['template']->getItems())) {
			$form['template']->setDefaultValue($this->entity->template);

			if ($this->httpRequest->getQuery('do') != 'templateBlockForm-loadInputs') {
				$this->templateReader->loadTemplateComponents($this['form']['component'], $this->httpRequest->getPost('template') ?: $this->entity->template);
				$this->templateReader->setDefaults($this['form']['component'], $this->entity->getTexts()->toArray(), $this->entity->template);
			}
		}
	}
}
