<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Pages\Model\Entities\TemplatePage;
use Pages\Model\Entities\TemplatePageText;
use Pages\Model\Paths;
use Pages\AdminModule\Model\TemplatePages;

class TemplatePageForm extends TemplateReaderControl
{
	/** @var TemplatePage */
	protected $entity;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Paths */
	protected $pathsService;

	public function __construct(TemplatePages $templatePages, Paths $paths)
	{
		$this->templatePagesService = $templatePages;
		$this->pathsService = $paths;
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir());
		$this->templateReader->setTranslateKey('templatePage');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addSelect('template', 'default.templates', $templates)->setPrompt('default.choose')->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplatePageText[] $texts */
			$texts = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->entity) {
				$entity = $this->entity;
				$texts = $entity->getTexts()->toArray();
			} else {
				$entity = new TemplatePage($values->title, $values->template);
			}

			if (!$langValues) {
				if ($this->getPresenter()->getAction() === 'add') {
					$tmp = $form->getHttpData()['component'];
					if (is_array($tmp)) {
						foreach ($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)) as $l => $v)
							$langValues[$l]['component'] = $v;
					}
				} else {
					foreach ($this->langsService->getLangs(false) as $l => $v) {
						$langValues[$l]['component'] = $componentValues;
					}
				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplatePageText($entity, $l);
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					$v[$name] = $componentValues[$name];
				}

				$texts[$l]->setTexts($v);
//				bdump($texts[$l]);
				$this->em->persist($texts[$l]);
			}

//			bdump($langValues);
//			bdump($componentValues);
//			die();

			$entity->setTexts($texts);

			$entity->title = $values->title;
			$entity->template = $values->template;

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity)->flush();
			$form->addCustomData('templatePageId', $entity->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setPage(int $id): void
	{
		$this->entity = $this->templatePagesService->get($id);

		if (!$this->entity)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form = $this['form'];
		$form->setDefaults([
			'title' => $this->entity->title,
		]);

		if ($this->entity->template && array_key_exists($this->entity->template, $form['template']->getItems())) {
			$form['template']->setDefaultValue($this->entity->template);

			if ($this->httpRequest->getQuery('do') != 'templatePageForm-loadInputs') {
				$this->templateReader->loadTemplateComponents($this['form']['component'], $this->httpRequest->getPost('template') ?: $this->entity->template);
				$this->templateReader->setDefaults($this['form']['component'], $this->entity->getTexts()->toArray(), $this->entity->template);
			}
		}
	}
}
