<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Pages\Model\Entities\TemplatePage;
use Pages\Model\Entities\TemplatePageText;
use Pages\AdminModule\Model\TemplatePages;

class TemplatePageForm extends TemplateReaderControl
{
	/** @var TemplatePage */
	protected $entity;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Sites */
	protected $sitesService;

	public function __construct(TemplatePages $templatePages, Sites $sites)
	{
		$this->templatePagesService = $templatePages;
		$this->sitesService         = $sites;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$httpRequest = $this->getPresenter()->getHttpRequest();

		$site = $httpRequest->getPost('site') ?: $this['form']['site']->getValue();
		$this->loadTemplates($site);

		$this['form']['site']->setHtmlAttribute('data-link-on-change', $this->link('loadTemplates!', '__val__'));
		$this['form']['template']->setHtmlAttribute('data-link-on-change', $this->link('loadInputs!', '__val__'));
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	public function handleLoadTemplates($site): void
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadTemplates($site);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('templates');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$sites = $this->sitesService->getOptionsForSelect();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		if (count($sites) > 1)
			$form->addSelect('site', 'default.site', $sites)
				->setRequired();
		else
			$form->addHidden('site', key($sites));
		$form->addSelect('template', 'default.templates', [])
			->setPrompt('default.choose')
			->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplatePageText[] $texts */
			$texts           = [];
			$componentValues = $form->getComponent('component')->getValues(true);


			if ($this->entity) {
				$entity = $this->entity;
				$texts  = $entity->getTexts()->toArray();
			} else {
				$entity = new TemplatePage($values->title, $values->template);
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
//				if ($this->getPresenter()->getAction() === 'add') {
//					$tmp = $form->getHttpData()['component'];
////					bdump($tmp);
////					bdump($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)));
//					if (is_array($tmp)) {
//						foreach ($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)) as $l => $v)
//							$langValues[$l]['component'] = $v;
//					}
//				} else {
//					foreach ($this->langsService->getLangs(false) as $l => $v) {
//						$langValues[$l]['component'] = $componentValues;
//					}
//				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplatePageText($entity, $l);
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					$v[$name] = $componentValues[$name];
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$entity->setTexts($texts);

			$entity->title    = $values->title;
			$entity->template = $values->template;
			$entity->site     = $this->sitesService->getReference($values->site);

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity)->flush();
			$form->addCustomData('templatePageId', $entity->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setPage(int $id): void
	{
		$this->entity = $this->templatePagesService->get($id);

		if (!$this->entity)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form = $this['form'];
		$d    = [
			'title' => $this->entity->title,
		];

		if ($this->entity->site) {
			$d['site'] = $this->entity->site->getIdent();
			$this->loadTemplates($this->entity->site->getIdent());
		}

		$form->setDefaults($d);

		$parsedTemplate = $this->parseTemplate($this->entity->template, $this->entity->site->getIdent());

		if (array_key_exists($parsedTemplate['template'], $form['template']->getItems())) {
			$form['template']->setDefaultValue($parsedTemplate['template']);

			if ($this->httpRequest->getQuery('do') != 'templatePageForm-loadInputs') {
				$parsedTemplate = $this->httpRequest->getPost('template')
					? $this->parseTemplate($this->httpRequest->getPost('template'), $this->entity->site->getIdent())
					: $parsedTemplate;

				$tmp = explode('|', $parsedTemplate['template']);
				if (count($tmp) == 2)
					$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($tmp[0]));


				$this->templateReader->loadTemplateComponents($this['form']['component'], $parsedTemplate['file']);
				$this->templateReader->setDefaults($this['form']['component'], $this->entity->getTexts()->toArray(), $parsedTemplate['file']);
			}
		}
	}

	protected function parseTemplate(string $template, string $siteIdent): array
	{
		$result = [
			'template' => $template,
			'file'     => $template,
		];

		if (strpos($template, '|') === false) {
			$key                = $siteIdent . '|' . $template;
			$result['template'] = isset($this['form']['template']->getItems()[$key]) ? $key : 'default' . '|' . $template;
		} else {
			$result['file'] = explode('|', $template)[1] ?? $template;
		}

		return $result;
	}

	protected function loadTemplates($site): void
	{
		$sites = $this->sitesService->getOptionsForSelect();
		if (!$site)
			$site = array_keys($sites)[0];

		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir());
		$this->templateReader->setTranslateKey('templatePage');
		$templates = [$this->t('default.default') => $this->templateReader->getTemplates('default')];

		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($site));
		$templates[$site] = $this->templateReader->getTemplates($site);

		$this['form']['template']->setItems($templates);
	}
}
