<?php declare(strict_types = 1);

namespace Pages\AdminModule\Presenters;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\ForbiddenRequestException;
use Nette\Application\UI\Multiplier;
use Nette\Http\IResponse;
use Pages\AdminModule\Components\ITemplatePageFormFactory;
use Pages\AdminModule\Components\ITemplatePagesGridFactory;
use Pages\AdminModule\Components\TemplatePageForm;
use Pages\AdminModule\Components\TemplatePagesGrid;
use Pages\Model\Entities\TemplatePage;

class TemplatePagesPresenter extends BasePresenter
{
	/** @var Sites @inject */
	public $sitesService;

	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('pages.title.templatePages'));
		$this->setHeader('pages.title.templatePages', 'fa fa-file-image');
	}

	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isAllowed($this->getFullModuleName(), 'accessTemplatePages')) {
			throw new ForbiddenRequestException();
		}
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$this->template->sites = $this->sitesService->getAll();
		$this['navigation']->setData(['header' => [
			[
				'title' => 'pages.menu.newTemplatePage',
				'link'  => 'TemplatePages:add',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAdd(): void
	{
		$this->setHeader('pages.title.newTemplatePage', 'fa fa-file-image');
	}

	public function actionEdit(int $id): void
	{
		$page = $this->em->getRepository(TemplatePage::class)->find($id);

		if (!$page)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader('pages.title.editTemplatePage', 'fa fa-file-image');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentTemplatePagesGrid(ITemplatePagesGridFactory $factory): Multiplier
	{
		return new Multiplier(function(string $sideIdent) use ($factory): TemplatePagesGrid {
			return $factory->create($sideIdent);
		});
	}

	protected function createComponentTemplatePageForm(ITemplatePageFormFactory $factory): TemplatePageForm
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setPage((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('edit', $form->getCustomData('templatePageId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function() {
			$this->redirect('default');
		};

		return $control;
	}
}
