<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\Parameters;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\BadRequestException;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Pages\AdminModule\Model\TemplateBlocks;
use Pages\Model\Entities\TemplateBlock;
use Pages\Model\Entities\TemplateBlockText;

class TemplateBlockForm extends TemplateReaderControl
{
	/** @var TemplateBlock|null */
	protected $entity;

	public function __construct(
		protected TemplateBlocks $templateBlocksService,
	)
	{
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir($this->pathsService->getTemplateBlocksDir());
		$this->templateReader->setTranslateKey('templateBlock');

		$form = $this->createForm();

		$templates[$this->t('default.default')] = $this->templateReader->getTemplates();

		foreach ((array) Parameters::load('system.templateBlocksDir') ?: [] as $k => $v) {
			$this->templateReader->setTemplatesDir($this->pathsService->parsePathParams($v));
			$templates[$this->t('admin.templateBlocksGroup.' . $k)] = $this->templateReader->getTemplates($k);
		}

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addSelect('template', 'default.templates', $templates)
			->setPrompt('default.choose')
			->setHtmlAttribute('data-confirm', $this->t('default.templateChangeConfirm'))
			->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();
		$form->addSaveCancelControl('saveControlSecondary')
			->setFloating();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplateBlockText[] $texts */
			$texts           = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->entity) {
				$entity = $this->entity;
				$texts  = $entity->texts->toArray();
			} else {
				$entity = new TemplateBlock($values->title, $values->template);
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
				//				if ($this->presenter->getAction() === 'add') {
				//					$tmp = $form->getHttpData()['component'];
				//					if (is_array($tmp)) {
				//						foreach ($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)) as $l => $v)
				//							$langValues[$l]['component'] = $v;
				//					}
				//				} else {
				//					foreach ($this->langsService->getLangs(false) as $l => $v) {
				//						$langValues[$l]['component'] = $componentValues;
				//					}
				//				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplateBlockText($entity, $l);
				}

				foreach ($v as $c => $cValue) {
					$comp = $form->getComponent('component')[$c];
					if ($comp instanceof BaseContainer === false) {
						continue;
					}

					$tmp = Json::decode(Json::encode($comp->getValues()), Json::FORCE_ARRAY);

					$v[$c] = array_merge($tmp, $cValue);

					if (array_key_exists('types', $tmp) && array_key_exists('types', $cValue)) {
						$v[$c]['types'] = array_merge($tmp['types'], $cValue['types']);
					}
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					if (is_array($componentValues[$name]) && array_key_exists($l, $componentValues[$name])) { // multilang chbox vracel pole s klicem jazyků
						$v[$name] = $componentValues[$name][$l];
					} else {
						$v[$name] = $componentValues[$name];
					}
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$entity->setTexts($texts);

			$entity->title    = $values->title;
			$entity->template = $values->template;

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity);
$this->em->flush();

			$cache = new Cache($this->cacheStorage, \Pages\FrontModule\Model\TemplateBlocks::CACHE_NAMESPACE);
			foreach ($this->langsService->getLangs(false) as $lang) {
				$cache->remove(\Pages\FrontModule\Model\TemplateBlocks::CACHE_NAMESPACE . '/' . $entity->getId() . '/' . $lang->getTag());
			}

			$form->addCustomData('templateBlockId', $entity->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */
	/**
	 * @throws BadRequestException
	 */
	public function setBlock(string|int $id): void
	{
		$this->entity = $this->templateBlocksService->get($id);

		if (!$this->entity) {
			$this->presenter->error();
		}

		$form = $this['form'];
		$form->setDefaults([
			'title' => $this->entity->title,
		]);

		if ($this->entity->template && array_key_exists(
				$this->entity->template,
				$form->getComponent('template')
					->getItems(),
			)) {
			$form->getComponent('template')->setDefaultValue($this->entity->template);

			$parsedTemplate = $this->parseTemplate($this->entity->template);

			if ($this->httpRequest->getQuery('do') !== 'templateBlockForm-loadInputs') {
				$parsedTemplate = $this->httpRequest->getPost('template')
					? $this->parseTemplate($this->httpRequest->getPost('template'))
					: $parsedTemplate;

				$tmp = explode('|', (string) $parsedTemplate['template']);
				if (count($tmp) === 2) {
					$this->templateReader->setTemplatesDir($this->pathsService->getTemplateBlocksDir($tmp[0]));
				}

				/** @var BaseContainer $container */
				$container = $this['form']['component'];
				$this->templateReader->loadTemplateComponents(
					$container,
					$parsedTemplate['file'],
					$this->entity->texts->toArray(),
				);
				$this->templateReader->setDefaults(
					$container,
					$this->entity->texts->toArray(),
					$parsedTemplate['file'],
				);
			}
		}
	}

	protected function parseTemplate(string $template, ?string $siteIdent = null): array
	{
		$result = [
			'template' => $template,
			'file'     => $template,
		];

		if (!str_contains($template, '|')) {
			$key                = $siteIdent . '|' . $template;
			$result['template'] = isset(
				$this['form']->getComponent('template')
					->getItems()[$key],
			) ? $key : 'default' . '|' . $template;
		} else {
			$result['file'] = explode('|', $template)[1] ?? $template;
		}

		return $result;
	}
}
