<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\AbortException;
use Nette\Utils\Html;
use Pages\AdminModule\Model\TemplatePageGroups;
use Pages\Model\Entities\TemplatePageGroup;

class TemplatePageGroupsGrid extends BaseControl
{
	public function __construct(
		protected TemplatePageGroups            $templatePageGroups,
		protected ITemplatePageGroupFormFactory $groupForm,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('pages.title.addGroup');
		$this->template->modal      = 'groupForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed|null $id
	 */
	public function handleEdit($id): void
	{
		$group = $this->templatePageGroups->get((int) $id);

		if (!$group) {
			return;
		}

		$this['groupForm']->setGroup((int) $id);
		$this->template->modalTitle = $this->t('pages.title.editGroup', null, ['title' => $group->title]);
		$this->template->modal      = 'groupForm';
		$this->redrawControl('modal');
	}

	/**
	 * @throws AbortException
	 */
	public function handleDelete(int|string $id): void
	{
		$presenter = $this->presenter;
		if ($this->templatePageGroups->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->templatePageGroups->getEr()->createQueryBuilder('tpg');

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('title', 'default.title')->setRenderer(fn(TemplatePageGroup $row) => Html::el('a', [
			'class' => 'ajax',
			'href'  => $this->link('edit!', $row->getId()),
		])->setText($row->title));

		// Filter
		$grid->addFilterText('title', 'default.title');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setConfirm('default.reallyDelete')
			->setBsType('danger')
			->addClass('ajax');

		return $grid;
	}

	public function createComponentGroupForm(): TemplatePageGroupForm
	{
		$control = $this->groupForm->create();
		if ($this->getParameter('id')) {
			$control->setGroup((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}
}
