<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Nette\Application\AbortException;
use Pages\AdminModule\Model\TemplatePageGroups;
use Pages\AdminModule\Model\TemplatePages;
use Pages\Model\Entities\TemplatePage;

class TemplatePagesGrid extends BaseControl
{
	public function __construct(
		protected string             $siteIdent,
		protected TemplatePages      $templatePagesService,
		protected TemplatePageGroups $templatePageGroups,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */
	/**
	 * @throws AbortException
	 */
	public function handleDelete(int|string $id): void
	{
		$presenter = $this->presenter;
		if ($this->templatePagesService->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->templatePagesService->getEr()->createQueryBuilder('p')
			->addSelect('g')
			->leftJoin('p.group', 'g')
			->andWhere('p.site = :site')
			->setParameters(new ArrayCollection([new Parameter('site', $this->siteIdent)]));

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('title', 'default.title', 'TemplatePages:edit');
		$grid->addColumnText('group', 'default.group')->setRenderer(fn(TemplatePage $row,
		) => $row->group ? $row->group->title : '');

		// Filter
		$grid->addFilterText('title', 'default.title');
		$grid->addFilterSelect(
			'group',
			'default.group',
			[null => ''] + $this->templatePageGroups->getOptionsForSelectGrouped(),
		)
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('g.id = :group')
					->setParameter('group', $value);
			});

		// Actions
		$grid->addAction('edit', '', 'TemplatePages:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setConfirm('default.reallyDelete')
			->setBsType('danger')
			->addClass('ajax');

		return $grid;
	}
}
