<?php declare(strict_types = 1);

namespace Pages\FrontModule\Presenters;

use Exception;
use Core\Model\Parameters;
use Pages\FrontModule\Model\Dao\DaoTemplatePage;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Helpers\TemplatePagesHelper;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	public ?DaoTemplatePage $templatePage = null;

	public function __construct(
		protected TemplatePages       $templatePagesService,
		protected Paths               $pathsService,
		protected TemplatePagesHelper $pagesHelper,
	)
	{

		parent::__construct();
	}

	protected function startup()
	{
		if ($this->getAction() === 'preview') {
			$this->skipSetInitial = true;
		}

		parent::startup();
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionTemplatePage(?int $id = null): void
	{
		$id   = $id ?: (int) $this->getActiveNavigation()->componentParams['templatePage'];
		$lang = $this->translator->getLocale();
		$page = $this->templatePagesService->get($id, $lang);

		if (!$page) {
			$this->error('pages.page.notFound');
		}

		$this->templatePage        = $page;
		$this->template->pageClass .= ' template-page tp-' . $page->getId();

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		if ($this->getParameter('pageOutput') === 'json') {
			$this->sendJson([
				'requestKey' => $this->getParameter('requestKey'),
				'texts'      => $texts,
			]);
		}

		$navLink = $this->getActiveNavigation()->link;
		if (is_array($navLink)) {
			$navLink = $navLink[0] ?? '';
		}
		$this['meta']->setMeta(
			'canonical',
			$this->getHttpRequest()
				->getUrl()
				->getBaseUrl() . ltrim($navLink ?: '/', '/'),
		);

		$this->template->templateTextValues = $texts;

		if (Parameters::load('system.useLatteTemplateReaderDefaultLang')) {
			$defaultLang = $this->translator->getDefaultLocale();

			$langPage = $this->templatePagesService->get($id, $defaultLang);
			if ($langPage) {
				$this->template->templateTextValuesLang[$defaultLang] = $langPage->getTexts();
			}
		}

		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
	}

	public function actionPreview(int $id, string $hash): void
	{
		$this->autoCanonicalize = false;
		$this->translator->setLocale($this->getParameter('locale'));
		$page = $this->templatePagesService->get($id);

		if (!$page || $page->getPreviewHash() !== $hash) {
			$this->error('pages.page.notFound');
		}

		$this->templatePage        = $page;
		$this->template->pageClass .= ' template-page tp-' . $page->getId();

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		$this['meta']->setMeta('robots', 'noindex,nofollow');

		$this->template->templateTextValues = $texts;
		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));

		try {
			ob_start();
			$this->template->render();
			ob_clean();
		} catch (Exception $e) {
			http_response_code(404);
			die();
		}
	}
}
