<?php declare(strict_types = 1);

namespace Pages\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TTPreload;
use Core\Model\Helpers\BaseFrontEntityService;
use Nette\Caching\Cache;
use Pages\FrontModule\Model\Dao\DaoTemplateBlock;
use Pages\Model\Entities\TemplateBlock;
use Throwable;

class TemplateBlocks extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'templateBlocks';

	protected $entityClass = TemplateBlock::class;

	protected EventDispatcher $eventDispatcher;

	protected array $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function __construct(
		EventDispatcher $eventDispatcher
	)
	{
		$this->eventDispatcher = $eventDispatcher;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @param string|int $id
	 *
	 * @throws Throwable
	 */
	public function get($id): ?DaoTemplateBlock
	{
		$lang = $this->translator->getLocale();
		$key  = self::CACHE_NAMESPACE . '/' . $id . '/' . $lang;

		$block = $this->getCache()->load($key, function(&$dep) use ($id, $lang) {
			$dep = $this->cacheDep;

			$templateBlock = $this->getEr()->createQueryBuilder('tb')
				->select('tb.id, tb.title, tbt.texts, tb.template')
				->join('tb.texts', 'tbt', 'WITH', 'tbt.lang = :lang')
				->where('tb.id = :id')
				->setParameters([
					'id'   => $id,
					'lang' => $lang,
				])
				->getQuery()
				->setMaxResults(1)
				->getOneOrNullResult();

			return $templateBlock ? $this->fillDao($templateBlock) : null;
		});

		if ($block) {
			$texts = $block->getTexts();
			$this->eventDispatcher->dispatch(new TTPreload($texts, [], []), 'core.ttPreload');
		}

		return $block;
	}

	protected function fillDao(array $data): DaoTemplateBlock
	{
		return new DaoTemplateBlock((int) $data['id'], $data['title'], $data['texts'], $data['template']);
	}
}
