<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplateBlocks;
use Pages\Model\Entities\TemplateBlock;
use Pages\Model\Entities\TemplateBlockText;

class TemplateBlockForm extends TemplateReaderControl
{
	/** @var TemplateBlock|null */
	protected $entity;

	public function __construct(protected TemplateBlocks $templateBlocksService)
	{
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir($this->pathsService->getTemplateBlocksDir());
		$this->templateReader->setTranslateKey('templateBlock');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addSelect('template', 'default.templates', $templates)
			->setPrompt('default.choose')
			->setHtmlAttribute('data-confirm', $this->t('default.templateChangeConfirm'))
			->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();
		$form->addSaveCancelControl('saveControlSecondary')
			->setFloating();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplateBlockText[] $texts */
			$texts           = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->entity) {
				$entity = $this->entity;
				$texts  = $entity->texts->toArray();
			} else {
				$entity = new TemplateBlock($values->title, $values->template);
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
				//				if ($this->presenter->getAction() === 'add') {
				//					$tmp = $form->getHttpData()['component'];
				//					if (is_array($tmp)) {
				//						foreach ($form->prepareMultilangTextsForEntity(array_keys($tmp), ArrayHash::from($tmp)) as $l => $v)
				//							$langValues[$l]['component'] = $v;
				//					}
				//				} else {
				//					foreach ($this->langsService->getLangs(false) as $l => $v) {
				//						$langValues[$l]['component'] = $componentValues;
				//					}
				//				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplateBlockText($entity, $l);
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					$v[$name] = $componentValues[$name];
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$entity->setTexts($texts);

			$entity->title    = $values->title;
			$entity->template = $values->template;

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity)->flush();
			$form->addCustomData('templateBlockId', $entity->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */
	/**
	 * @throws BadRequestException
	 */
	public function setBlock(string|int $id): void
	{
		$this->entity = $this->templateBlocksService->get($id);

		if (!$this->entity) {
			$this->presenter->error();
		}

		$form = $this['form'];
		$form->setDefaults([
			'title' => $this->entity->title,
		]);

		if ($this->entity->template && array_key_exists(
				$this->entity->template,
				$form->getComponent('template')
					->getItems(),
			)) {
			$form->getComponent('template')->setDefaultValue($this->entity->template);

			if ($this->httpRequest->getQuery('do') !== 'templateBlockForm-loadInputs') {
				/** @var BaseContainer $container */
				$container = $this['form']['component'];
				$this->templateReader->loadTemplateComponents(
					$container,
					$this->httpRequest->getPost('template') ?: $this->entity->template,
					$this->entity->texts->toArray(),
				);
				$this->templateReader->setDefaults(
					$container,
					$this->entity->texts->toArray(),
					$this->entity->template,
				);
			}
		}
	}
}
