<?php declare(strict_types = 1);

namespace Pages\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\UI\InvalidLinkException;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Pages\AdminModule\Model\TemplatePageGroups;
use Pages\AdminModule\Model\TemplatePages;
use Pages\Model\Entities\TemplatePage;
use Pages\Model\Entities\TemplatePageText;

class TemplatePageForm extends TemplateReaderControl
{
	/** @var TemplatePage|null */
	protected $entity;

	public function __construct(
		protected TemplatePages      $templatePagesService,
		protected Sites              $sitesService,
		protected TemplatePageGroups $templatePageGroups,
	)
	{
	}

	/**
	 * @param IComponent $presenter
	 *
	 * @throws InvalidLinkException
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$httpRequest = $this->presenter->getHttpRequest();

		$site = $httpRequest->getPost('site') ?: $this['form']->getComponent('site')->getValue();
		$this->loadTemplates($site);

		$this['form']->getComponent('site')
			->setHtmlAttribute('data-link-on-change', $this->link('loadTemplates!', '__val__'));
		$this['form']->getComponent('template')
			->setHtmlAttribute('data-link-on-change', $this->link('loadInputs!', '__val__'));
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param string $site
	 */
	public function handleLoadTemplates($site): void
	{
		$presenter = $this->presenter;

		try {
			$this->loadTemplates($site);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (Exception) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('templates');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$sites = $this->sitesService->getOptionsForSelect();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);

		$form->addSelect(
			'group',
			'default.group',
			[null => $this->t('default.choose')] + $this->templatePageGroups->getOptionsForSelectGrouped(),
		)
			->setTranslator(null);

		if (count($sites) > 1) {
			$form->addSelect('site', 'default.site', $sites)
				->setRequired();
		} else {
			$form->addHidden('site', key($sites));
		}

		$form->addSelect('template', 'default.templates', [])
			->setTranslator(null)
			->setPrompt($this->t('default.choose'))
			->setHtmlAttribute('data-confirm', $this->t('default.templateChangeConfirm'))
			->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();
		$form->addSaveCancelControl('saveControlSecondary')
			->setFloating();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TemplatePageText[] $texts */
			$texts           = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->entity) {
				$entity = $this->entity;
				$texts  = $entity->texts->toArray();
			} else {
				$entity = new TemplatePage($values->title, $values->template);
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					$texts[$l] = new TemplatePageText($entity, $l);
				}

				foreach ($v as $c => $cValue) {
					$comp = $form['component'][$c];
					if ($comp instanceof BaseContainer === false) {
						continue;
					}

					$tmp = Json::decode(Json::encode($comp->getValues()), Json::FORCE_ARRAY);

					$v[$c] = array_merge($tmp, $cValue);
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					if (is_array($componentValues[$name]) && array_key_exists(
							$l,
							$componentValues[$name],
						)) { // multilang chbox vracel pole s klicem jazyků
						$v[$name] = $componentValues[$name][$l];
					} else {
						$v[$name] = $componentValues[$name];
					}
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$entity->setTexts($texts);

			$entity->title    = $values->title;
			$entity->template = $values->template;
			$entity->site     = $this->sitesService->getReference($values->site);

			$group         = $values->group ? $this->templatePageGroups->getReference($values->group) : null;
			$entity->group = $group;

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->persist($entity)->flush();
			$form->addCustomData('templatePageId', $entity->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setPage(int $id): void
	{
		$this->entity = $this->templatePagesService->get($id);

		if (!$this->entity) {
			$this->presenter->error();
		}

		$form = $this['form'];
		$d    = [
			'title' => $this->entity->title,
			'group' => $this->entity->group ? $this->entity->group->getId() : null,
		];

		if ($this->entity->site) {
			$d['site'] = $this->entity->site->getIdent();
			$this->loadTemplates($this->entity->site->getIdent());
		}

		$form->setDefaults($d);

		$parsedTemplate = $this->parseTemplate($this->entity->template, $this->entity->site->getIdent());

		if (array_key_exists($parsedTemplate['template'], $form->getComponent('template')->getItems())) {
			$form->getComponent('template')->setDefaultValue($parsedTemplate['template']);

			if ($this->httpRequest->getQuery('do') != 'templatePageForm-loadInputs') {
				$parsedTemplate = $this->httpRequest->getPost('template')
					? $this->parseTemplate($this->httpRequest->getPost('template'), $this->entity->site->getIdent())
					: $parsedTemplate;

				$tmp = explode('|', (string) $parsedTemplate['template']);
				if (count($tmp) == 2) {
					$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($tmp[0]));
				}

				/** @var BaseContainer $container */
				$container = $this['form']['component'];
				$this->templateReader->loadTemplateComponents(
					$container,
					$parsedTemplate['file'],
					$this->entity->texts->toArray(),
				);
				$this->templateReader->setDefaults(
					$container,
					$this->entity->texts->toArray(),
					$parsedTemplate['file'],
				);
			}
		}
	}

	protected function parseTemplate(string $template, string $siteIdent): array
	{
		$result = [
			'template' => $template,
			'file'     => $template,
		];

		if (!str_contains($template, '|')) {
			$key                = $siteIdent . '|' . $template;
			$result['template'] = isset(
				$this['form']->getComponent('template')
					->getItems()[$key],
			) ? $key : 'default' . '|' . $template;
		} else {
			$result['file'] = explode('|', $template)[1] ?? $template;
		}

		return $result;
	}

	protected function loadTemplates(?string $site = null): void
	{
		$sites = $this->sitesService->getOptionsForSelect();
		if (!$site) {
			$site = array_keys($sites)[0];
		}

		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir());
		$this->templateReader->setTranslateKey('templatePage');
		$templates = [$this->t('default.default') => $this->templateReader->getTemplates('default')];

		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($site));
		$templates[$site] = $this->templateReader->getTemplates($site);

		$this['form']->getComponent('template')->setItems($templates);
	}
}
