<?php declare(strict_types = 1);

namespace Pages\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TTPreload;
use Core\Model\Helpers\BaseFrontEntityService;
use Nette\Caching\Cache;
use Pages\FrontModule\Model\Dao\DaoTemplatePage;
use Pages\Model\Entities\TemplatePage;
use Pages\Model\Paths;

class TemplatePages extends BaseFrontEntityService
{
	final public const CACHE_NAMESPACE = 'templatePages';

	protected $entityClass = TemplatePage::class;

	protected array $cacheDep = [
		Cache::Tags   => [self::CACHE_NAMESPACE],
		Cache::Expire => '1 week',
	];

	public function __construct(protected Paths $paths, protected EventDispatcher $eventDispatcher)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function get(int $id): ?DaoTemplatePage
	{
		$lang = $this->translator->getLocale();
		$key  = 'templatePage/' . $id . '/' . $lang;

		/** @var null|DaoTemplatePage $page */
		$page = $this->getCache()->load($key, function(&$dep) use ($id, $lang) {
			$dep = $this->cacheDep;

			$templatePage = $this->getEr()->createQueryBuilder('tp')
				->select('tp.id, tp.title, tpt.texts, tp.template')
				->leftJoin('tp.texts', 'tpt', 'WITH', 'tpt.lang = :lang')
				->where('tp.id = :id')
				->setParameters([
					'id'   => $id,
					'lang' => $lang,
				])
				->getQuery()
				->setMaxResults(1)
				->getOneOrNullResult();

			return $templatePage ? $this->fillDao($templatePage) : null;
		});

		if ($page) {
			$texts = $page->getTexts();
			$this->eventDispatcher->dispatch(new TTPreload($texts, [], []), 'core.ttPreload');
		}

		return $page;
	}

	public function getTemplatePathByFile(string $file): string
	{
		return $this->paths->getTemplatePagesFile($file);
	}

	public function getTemplatePathById(int $id): ?string
	{
		$template = $this->get($id);

		return $template ? $this->paths->getTemplatePagesFile($template->getTemplate()) : null;
	}

	protected function fillDao(array $data): DaoTemplatePage
	{
		return new DaoTemplatePage((int) $data['id'], $data['title'], $data['texts'] ?: [], $data['template']);
	}
}
