<?php declare(strict_types = 1);

namespace PayPal\FrontModule\Model\Subscribers;

use Brick\Money\Money;
use PayPal\FrontModule\Components\PaypalControl;
use PayPal\FrontModule\Components\IPaypalControlFactory;
use PayPal\FrontModule\Model\Paypal;
use PayPal\FrontModule\Model\Exceptions\PaypalException;
use PayPal\FrontModule\Service\PaymentService;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\PresenterTemplateEvent;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\FrontModule\Presenters\PaymentPresenter;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Orders;
//use Gls\Model\Exception;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderPaymentSubscriber implements EventSubscriberInterface
{
	protected IPaypalControlFactory $paypalControlFactory;
	protected CardsPaymentService $cardsPaymentService;
    protected PaymentService $paymentService;
	protected EntityManagerDecorator $em;
	protected Orders $orders;
    protected Paypal $paypal;

	public function __construct(IPaypalControlFactory $paypalControlFactory, CardsPaymentService $cardsPaymentService,
                                Orders $orders, EntityManagerDecorator $em, Paypal $paypal)
	{
		$this->paypalControlFactory  = $paypalControlFactory;
		$this->cardsPaymentService   = $cardsPaymentService;
        $this->paymentService        = new PaymentService();
		$this->orders                = $orders;
		$this->em                    = $em;
        $this->paypal                = $paypal;
	}

	/**
	 * @return string[]
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderFinishedRender'   => 'orderFinishedRender',
			'eshopOrders.paymentPayAction'      => 'paymentPayAction',
			'eshopOrders.paymentFinishedRender' => 'paymentFinishedRender',
		];
	}

	/**
	 * @param PresenterTemplateEvent $event
	 */
	public function orderFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var FinishedPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;

		if(! $this->checkPaymentType($order)) return;

		$presenter->template->link = $presenter->link('Payment:pay', ['orderIdent' => $order->getIdent(), 'activeNavigation' => null]);
		$presenter->addIncludeTemplate(__DIR__ . '/PaymentButton.latte');
	}

	public function paymentPayAction(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;
		$self      = $this;

        if(! $this->checkPaymentType($order)) return;

        $control = $this->paypalControlFactory->create($this->paymentService, $this->paypal);
        $presenter->addComponent($control, 'paypalButton');
        //var_dump($presenter->link('EshopOrders:Front:Payment:paymentFinished', 1));

        $token = $this->cardsPaymentService->getLastCreated($order->getId()) ?: $this->cardsPaymentService->createToken($order);
        //var_dump($presenter->getHttpRequest()->getQuery('oid'));

        $id = $control->createPayment($order);
        if($id) {
            $token->param = $id;
            $self->em->persist($token);
            $self->em->flush($token);
            $self->cardsPaymentService->tokenCheckout($token->getId());

            $presenter->template->paypalOrderID = $token->param;
            $presenter->addIncludeTemplate(__DIR__ . '/PaypalButton.latte');

            //$presenter->template->confirmLink = $control->getConfirmLink();
            $presenter->template->sdkLink = $this->paypal->getSDKURL($order->getCurrencyCode());
        }


		/*$control->onCheckout[] = static function(ComgateControl $control, Payment $payment, $data) use ($token, $self) {
			$token->param = $data['transId'];
			$self->em->persist($token);
			$self->em->flush($token);
			$self->cardsPaymentService->tokenCheckout($token->getId());
		};*/

		//$control->handleCheckout();
	}

	/**
	 * @param PresenterTemplateEvent $event
	 */
	public function paymentFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$self      = $this;
		$params    = $presenter->getParameters();

		$oid        = $params['oid'] ?: $presenter->getHttpRequest()->getPost('oid');
		$orderIdent = $params['orderIdent'] ?: $presenter->getHttpRequest()->getPost('orderIdent');
		$status     = null;
		$msg        = null;
		Debugger::log($presenter->getHttpRequest()->getQuery(), '_paypal-full-payment-finished');

		if ($oid && $orderIdent) {
			$order = $this->orders->getByIdent($orderIdent);

			if(! $this->checkPaymentType($order)) return;

			$presenter->order           = $order;
			$presenter->template->order = $order;

            if(! $order->isPaid) {

                $token    = $this->cardsPaymentService->getLastCreated($order->getId());
                $control  = $this->paypalControlFactory->create($this->paymentService, $this->paypal);
                $data     = $control->capturePayment($oid);
                Debugger::log($data, '_paypal-full-response');

                if(isset($data['status']) && ($data['status'] == 'COMPLETED')) {
                    if ($token) $self->cardsPaymentService->tokenPaid($token->getToken());
                } else {
                    $msg = isset($data['name']) ? $data['name'] : 'Unknown error';
                    $self->cardsPaymentService->tokenError($token, $msg);
                    $presenter->template->cardError = $msg;
                }
            }

            $presenter->addIncludeTemplate(__DIR__ . '/PaymentFinished.latte');
		}
	}

    private function checkPaymentType($order): bool
    {
		if (!$order || $order->getPayment()->getPayment()->getIdent() !== 'card' || $order->getPayment()->getPayment()->code1 !== 'paypal') {
			return false;
		}

        return true;
    }

}
