<?php declare(strict_types = 1);

namespace Ppl\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Ppl\Model\Entities\PplOrder;
use Ppl\Model\Entities\PplParcelNumber;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\ControlEvent;
use Ppl\Model\OrdersExported;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	protected OrdersExported         $ordersExported;
	protected EntityManagerDecorator $em;
	protected EventDispatcher        $eventDispatcher;

	/** @var PplOrder[]|null */
	protected static ?array $allOrders = null;

	public function __construct(OrdersExported $ordersExported, EntityManagerDecorator $em, EventDispatcher $eventDispatcher)
	{
		$this->ordersExported  = $ordersExported;
		$this->em              = $em;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::createForm'    => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'      => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'ppl') {
			$control->template->pplExport                   = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data   = $event->data;
		$export = null;

		/** @var Order $order */
		$order = $data['order'];
		/** @var Html $html */
		$html = $data['html'];

		if ($data['allIds']) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported($data['allIds']);
			}
			$export = self::$allOrders[$order->getId()] ?? null;
		} else {
			$export = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setHref($url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$pplIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'ppl') {
				$pplIds[] = $sped->getId();
			}
		}

		if (empty($pplIds)) {
			return;
		}

		$container = $form->addContainer('ppl', '');
		$container->addText('numberPackage', 'ppl.entity.numberPackage')
			->setDescription('ppl.entity.numberPackageDesc');
		$container->addBool('removeExport', 'ppl.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'ppl.entity.updateDeliveryAddress')
			->setDefaultValue(0);

		$form->getComponent('spedition')->addCondition($form::IS_IN, $pplIds)
			->toggle($form['ppl']['numberPackage']->getHtmlId())
			->toggle($form['ppl']['removeExport']->getHtmlId())
			->toggle($form['ppl']['updateDeliveryAddress']->getHtmlId());
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control   = $event->control;
		$orderId   = $orderSpedition->getOrder()->getId();
		$values    = $event->values;
		$spedition = null;
		$pplOrder  = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isPpl = false;
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'ppl' && $sped->getId() == $values->spedition) {
				$isPpl = true;
				break;
			}
		}

		try {
			if ($isPpl) {
				if (!$pplOrder) {
					$pplOrder = new PplOrder($orderSpedition->getOrder());
				}

				$numberPackages      = trim(str_replace(' ', '', (string) $values->ppl->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && Strings::contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);

				if ($numberPackagesCount <= 1) {

					$pplOrder->numberPackage = $numberPackages[0] ?? null;
					$pplOrder->associatedNumberPackages->clear();
				} else {

					$existsPackages            = $pplOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$pplOrder->numberPackage = $numberPackage;
						} else {
							if (in_array($numberPackage, $diffToAdd)) {
								$pplOrder->associatedNumberPackages->add(new PplParcelNumber($numberPackage, $pplOrder));
							}
						}
					}

					foreach ($diffToRemove as $p) {
						$pplOrder->associatedNumberPackages->remove($p);
					}
				}

				if ($pplOrder->numberPackage && $pplOrder->getExported() === null) {
					$pplOrder->export();
				}

				if ($values->ppl->removeExport === 1) {
					$pplOrder->resetExport();
				}

				$this->em->persist($pplOrder);

				if ($event->presenter && $values['ppl']->updateDeliveryAddress) {
					$event->presenter->flashMessageWarning('ppl.orderForm.updateDeliveryAddress');
					$event->presenter->redrawControl('flashes');
				}

				$this->eventDispatcher->dispatch(new Event(['order'          => $pplOrder->getOrder(),
				                                            'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($pplOrder) {
				$this->em->remove($pplOrder);
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('ppl.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$pplOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$pplOrder) {
			return;
		}

		$event->form->getComponent('ppl')->setDefaults([
			'numberPackage' => implode(', ', $pplOrder->getAllNumberPackages()),
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'ppl')) {
			return;
		}

		$orderId  = $order->getId();
		$pplOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$pplOrder) {
			return;
		}

		$pplOrder->resetExport();
		$this->em->persist($pplOrder);
		$this->em->flush($pplOrder);
	}
}
