<?php declare(strict_types = 1);

namespace Ppl\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\OrderAddress;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Nette\Utils\JsonException;
use Ppl\Model\Entities\PplOrder;
use Ppl\Model\Entities\PplParcelNumber;
use Salamek\PplMyApi\Api;
use Salamek\PplMyApi\Enum\Country;
use Salamek\PplMyApi\Enum\LabelDecomposition;
use Salamek\PplMyApi\Exception\OfflineException;
use Salamek\PplMyApi\Exception\SecurityException;
use Salamek\PplMyApi\Exception\WrongDataException;
use Salamek\PplMyApi\Model\CityRouting;
use Salamek\PplMyApi\Model\Flag;
use Salamek\PplMyApi\Model\IPackageService;
use Salamek\PplMyApi\Model\ISpecialDelivery;
use Salamek\PplMyApi\Model\Package;
use Salamek\PplMyApi\Model\PackageSet;
use Salamek\PplMyApi\Model\PaymentInfo;
use Salamek\PplMyApi\Model\Recipient;
use Salamek\PplMyApi\Model\Sender;
use Salamek\PplMyApi\Model\SpecialDelivery;
use Salamek\PplMyApi\PdfLabel;
use Tracy\Debugger;

class ApiService
{
	protected EntityManagerDecorator $em;
	protected Sellers                $sellers;

	protected static ?int $currentSeriesNumber = null;
	protected ?array      $lastNumbersFromDb   = null;
	protected string      $pplLabelsDir        = DATA_DIR . '/ppl/labels';

	public function __construct(EntityManagerDecorator $em, Sellers $sellers)
	{
		$this->em      = $em;
		$this->sellers = $sellers;
	}

	/**
	 * @param PplOrder[] $orders
	 *
	 * @throws Exception
	 */
	public function sendOrders(array $orders, int $quantity = 1): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
		];

		$pplMyApi = new Api(
			PplConfig::load('username'),
			PplConfig::load('password'),
			PplConfig::load('customerId'),
		);

		foreach ($orders as $pplOrder) {
			$order      = $pplOrder->getOrder();
			$pplProduct = $pplOrder->pplProduct;

			if ($order->getPaymentIdent() === 'cod') {
				$pplProduct = $pplOrder->pplProductCod;
			}

			$seller = $this->sellers->getSellerForSite($order->site->getIdent());

			$recipient = $this->getRecipient($order->getAddressDelivery());
			$sender    = $this->getSender($seller);

			/** @phpstan-ignore-next-line */
			$cityRoutingResponse = $pplMyApi->getCitiesRouting($recipient->getCountry(), null, $recipient->getZipCode(), $recipient->getStreet());

			if (is_array($cityRoutingResponse)) {
				$cityRoutingResponse = $cityRoutingResponse[0];
			}

			if (!isset($cityRoutingResponse->RouteCode) || !isset($cityRoutingResponse->DepoCode) || !isset($cityRoutingResponse->Highlighted)) {
				$logMessage = $order->getId() . ' - Export PPL se nezdařil, chybí Routing, pravděpodobně neplatná adresa!';
				Debugger::log($logMessage, 'ppl');
				LogNotifier::toDevelopers($logMessage, 'PPL - ' . $order->site->getIdent());
				$result['error']++;

				continue;
			}

			$cityRouting = new CityRouting(
				$cityRoutingResponse->RouteCode,
				$cityRoutingResponse->DepoCode,
				$cityRoutingResponse->Highlighted
			);

			/** @var Package[] $packages */
			$packages   = [];
			$packageSet = null;
			for ($i = 1; $i <= $quantity; $i++) {
				$packageNumber = $this->getPackageSeriesNumber($pplProduct);

				$parcelNumber = new PplParcelNumber((string) $packageNumber, $pplOrder);

				$this->em->persist($parcelNumber);

				$package = new Package((string) $packageNumber, $pplProduct, '', $recipient, $cityRouting);
				$package->setFlags([
					new Flag(\Salamek\PplMyApi\Enum\Flag::SMART_LABEL, true),
				]);

				if ($order->getPaymentIdent() === 'cod' && $i === 1) {
					$cod         = round($order->getPrice(true), $order->currency->decimals ?? 0);
					$paymentInfo = new PaymentInfo($cod, $order->getCurrencyCode(), $order->getId());

					$package->setPaymentInfo($paymentInfo);
				}

				if ($pplOrder->pointId) {
					$package->setSpecialDelivery(new SpecialDelivery($pplOrder->pointId));
				}

				if ($i === 1) {
					$pplOrder->numberPackage = (string) $packageNumber;
					$this->em->persist($pplOrder);

					$packageSet = new PackageSet((string) $packageNumber, 1, $quantity);
				} else {
					$packageSet      = clone $packageSet;
					$packagePosition = $packageSet->getPackagePosition() + 1;
					$packageSet->setPackagePosition($packagePosition);
				}

				$package->setPackageSet($packageSet);

				$packages[] = $package;
			}

			$r = $pplMyApi->createPackages($packages);
			if (!is_array($r)) {
				$r = [$r];
			}

			foreach ($r as $v) {
				if ($v->Code === '0') {
					Debugger::log('create packages ' . $order->getId(), 'ppl');
					$result['ok']++;

					$pplOrder->export();
					$this->em->persist($pplOrder);
				} else {
					Debugger::log('ERROR: ' . Json::encode($v), 'ppl');
					$result['error']++;
				}
			}

			if ($result['ok'] > 0) {
				foreach ($packages as &$package) {
					$package->setSender($sender);
				}
				FileSystem::createDir($this->pplLabelsDir);
				$rawPdf = PdfLabel::generateLabels($packages);
				file_put_contents($this->getPplLabelFile($order->getId()), $rawPdf);
			}

			$this->em->flush();
		}

		return $result;
	}

	public function getLastPackageNumbersFromDb(int $service, int $min, int $max): array
	{
		if ($this->lastNumbersFromDb === null) {
			$this->lastNumbersFromDb = [];
			$date                    = (new DateTime)->modify('-1 year');

			foreach ($this->em->getRepository(PplParcelNumber::class)->createQueryBuilder('n')
				         ->select('MAX(n.numberPackage) as max, p.pplProduct')
				         ->andWhere('n.numberPackage BETWEEN :min AND :max')
				         ->innerJoin(PplOrder::class, 'p', Join::WITH, 'p.pplProduct = :service AND p.exported > :date')
				         ->setParameters([
					         'service' => $service,
					         'min'     => $min,
					         'max'     => $max,
					         'date'    => $date,
				         ])
				         ->groupBy('p.pplProduct')
				         ->getQuery()->getArrayResult() as $row) {
				$this->lastNumbersFromDb[$row['pplProduct']] = (int) $row['max'];
			}

			foreach ($this->em->getRepository(PplParcelNumber::class)->createQueryBuilder('n')
				         ->select('MAX(n.numberPackage) as max, p.pplProductCod')
				         ->andWhere('n.numberPackage BETWEEN :min AND :max')
				         ->innerJoin(PplOrder::class, 'p', Join::WITH, 'p.pplProductCod = :service AND p.exported > :date')
				         ->setParameters([
					         'service' => $service,
					         'min'     => $min,
					         'max'     => $max,
					         'date'    => $date,
				         ])
				         ->groupBy('p.pplProductCod')
				         ->getQuery()->getArrayResult() as $row) {
				$this->lastNumbersFromDb[$row['pplProductCod']] = (int) $row['max'];
			}
		}

		return $this->lastNumbersFromDb;
	}

	/**
	 * @param PplOrder[] $orders
	 */
	public function generateLabels(array $orders): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		foreach ($orders as $pplOrder) {
			$order = $pplOrder->getOrder();
			$file  = $this->getPplLabelFile($order->getId());

			if (file_exists($file)) {
				$result['files'][] = $file;
				$result['ok']++;
			} else {
				$result['error']++;
			}
		}

		$this->em->flush();

		return $result;
	}

	protected function getPackageSeriesNumber(int $service): ?int
	{
		$list = PplConfig::load('packageNumbers')[$service];

		if ($list) {
			if (self::$currentSeriesNumber === null) {
				$lastFromDb = $this->getLastPackageNumbersFromDb($service, $list[0], $list[1]);
				if (isset($lastFromDb[$service])) {
					self::$currentSeriesNumber = $lastFromDb[$service] + 1;
				} else {
					self::$currentSeriesNumber = $list[0];
				}
			} else {
				self::$currentSeriesNumber++;
			}

			if (self::$currentSeriesNumber > $list[1]) {
				self::$currentSeriesNumber = $list[0];
			}

			return self::$currentSeriesNumber;
		}

		return null;
	}

	protected function getSender(Seller $seller): Sender
	{
		return new Sender(
			$seller->city,
			$seller->name,
			$seller->address,
			$seller->postal,
			$seller->email,
			$seller->phone,
			'',
			$seller->country ? Strings::upper($seller->country) : Country::CZ,
		);
	}

	protected function getRecipient(OrderAddress $address): Recipient
	{
		return new Recipient(
			$address->getCity(),
			Strings::truncate($address->getName(), 50, ''),
			$address->getStreet(),
			$address->getPostal(),
			$address->getEmail(),
			$address->getPhone(),
			'',
			$address->getCountry() ? Strings::upper($address->getCountry()->getId()) : Country::CZ,
			$address->getCompany(),
		);
	}

	protected function getPplLabelFile(int $orderId): string
	{
		return $this->pplLabelsDir . '/' . $orderId . '.pdf';
	}

}
