<?php declare(strict_types = 1);

namespace Ppl\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use Nette\Utils\DateTime;
use Ppl\Model\PplConfig;
use Salamek\PplMyApi\Enum\Product;

/**
 * @ORM\Table(name="ppl__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopOrders\Model\Listeners\CarrierListener"})
 */
class PplOrder
{
	public const STATUS_COMPLETED = 'completed';
	public const STATUS_NOT_FOUND = 'notFound';
	public const STATUS_RETURNED  = 'returned';

	/**
	 * @ORM\Id
	 * @ORM\OneToOne(targetEntity="\EshopOrders\Model\Entities\Order")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Order $order;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private ?DateTimeInterface $exported = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $numberPackage = null;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"unsigned": true})
	 */
	public int $pplProduct = Product::PPL_PARCEL_CZ_PRIVATE;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"unsigned": true})
	 */
	public int $pplProductCod = Product::PPL_PARCEL_CZ_PRIVATE_COD;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $lastStatus = null;

	/**
	 * @ORM\Column(name="point_id", type="string", nullable=true)
	 */
	public ?string $pointId = null;

	/**
	 * @var Collection<string, PplParcelNumber>
	 * @ORM\OneToMany(targetEntity="PplParcelNumber", mappedBy="parcelOrder", cascade={"all"}, orphanRemoval=true, indexBy="numberPackage")
	 */
	public Collection $associatedNumberPackages;

	public function __construct(Order $order)
	{
		$this->order                    = $order;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function export(?DateTime $datetime = null): self
	{
		$this->exported = $datetime ?? new DateTime;

		return $this;
	}

	public function getExported(): ?DateTimeInterface { return $this->exported; }

	public function getTrackingUrl(): string
	{
		return sprintf(PplConfig::load('trackingUrl', ''), $this->numberPackage);
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(PplConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->lastStatus    = null;
		$this->exported      = null;
		$this->associatedNumberPackages->clear();
	}

	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(PplParcelNumber $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}

}
