<?php declare(strict_types = 1);

namespace Ppl\Model;

use Core\Model\Helpers\BaseEntityService;
use Ppl\Model\Entities\PplOrder;

/**
 * @method PplOrder|null getReference($id)
 * @method PplOrder[] getAll()
 * @method PplOrder|null get($id)
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = PplOrder::class;

	/**
	 * @param int[]|string[] $ids
	 * @return PplOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @param int[]|string[] $ids
	 * @return PplOrder[]
	 */
	public function getOrdersExported(array $ids = []): array
	{
		return $this->getOrders(true, $ids);
	}

	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->getEr()->createQueryBuilder('go')
			         ->where('go.lastStatus NOT IN (:status) OR go.lastStatus IS NULL')
			         ->andWhere('go.numberPackage IS NOT NULL')
			         ->setParameters([
				         'status' => [
					         PplOrder::STATUS_COMPLETED,
					         PplOrder::STATUS_NOT_FOUND,
					         PplOrder::STATUS_RETURNED,
				         ],
			         ])->getQuery()
			         ->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $ids
	 * @return PplOrder[]
	 */
	public function getOrders(?bool $isExported = null, array $ids = []): array
	{
		$qb = $this->getEr()->createQueryBuilder('go')
			->select('o, go, oad, oai, os, s')
			->join('go.order', 'o')
			->leftJoin('o.addressDelivery', 'oad')
			->leftJoin('o.addressInvoice', 'oai')
			->leftJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 's')
			->orderBy('o.id')->groupBy('o.id');

		if ($isExported === true) {
			$qb->where('go.exported IS NOT NULL');
		} else if ($isExported === false) {
			$qb->where('go.exported IS NULL');
		}

		if (!empty($ids)) {
			$qb->andWhere('go.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	public function markAsCompleted(array $orderIds): void
	{
		foreach ($this->getEr()->createQueryBuilder('go')
			         ->where('go.order IN (:ids)')
			         ->setParameter('ids', $orderIds)
			         ->getQuery()->getResult() as $row) {
			/** @var PplOrder $row */
			$row->lastStatus = PplOrder::STATUS_COMPLETED;
			$this->em->persist($row);
		}

		$this->em->flush();
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->getEr()->createQueryBuilder('g')
			         ->select('IDENTITY(g.order) as id')
					 ->leftJoin('g.associatedNumberPackages', 'acp')
					 ->orWhere('g.numberPackage LIKE :number')
					 ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
