<?php declare(strict_types = 1);

namespace Ppl\Model\Subscribers;

use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Dao\PaymentSpedition;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Http\Request;
use Nette\Localization\Translator;
use Ppl\Model\Entities\PplOrder;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected PaymentSpeditions      $paymentSpeditions;
	protected Translator             $translator;
	protected Request                $httpRequest;
	protected Countries              $countries;

	public function __construct(
		EntityManagerDecorator $em,
		PaymentSpeditions      $paymentSpeditions,
		Translator             $translator,
		Request                $request,
		Countries              $countries
	)
	{
		$this->em                = $em;
		$this->paymentSpeditions = $paymentSpeditions;
		$this->translator        = $translator;
		$this->httpRequest       = $request;
		$this->countries         = $countries;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form    = $event->form;
		$methods = $this->getPplSpeditionIds();

		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);
		$container = $form->addContainer('pplPickup');

		foreach ($methods as $row) {
			/** @var Spedition $spedition */
			$spedition = $row->getSpedition();
			if (!$spedition->isPickup) {
				continue;
			}

			$speditionId = $spedition->getId();

			foreach ($row->getCountries() as $country) {
				$k = $country->getId() . '_' . $speditionId;
				$container->addHidden($k);
				foreach (['street', 'city', 'postal'] as $col) {
					$container->addHidden($k . '_' . $col);
				}
				$container->addText($k . '_p')
					->setHtmlAttribute('data-ppl-choose-pickup-point')
					->setPlaceholder('pplFront.orderForm.chooseAddress')
					->setHtmlAttribute('data-country', $country->getId())
					->setHtmlAttribute('data-target', $container->getComponent($k)->getHtmlId());

				foreach (['street', 'city', 'postal'] as $col) {
					$container->getComponent($k . '_p')
						->setHtmlAttribute('data-' . $col, $container->getComponent($k . '_' . $col)->getHtmlId());
				}

				/** @phpstan-ignore-next-line */
				$event->template->pplLastKey = $k;
			}
		}
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data = $event->data;
		$form = $event->form;

		foreach ($data['pplPickup'] ?? [] as $k => $v) {
			if (isset($form->getComponent('pplPickup', false)[$k])) {
				$form->getComponent('pplPickup')[$k]->setDefaultValue($v);
			}
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;
		$post    = $request->getPost();

		$spedition = $this->getSpeditions()[$data['speditionCountry']][$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];

			$data['pplPickup'][$k] = $request->getPost('pplPickup')[$k];
			foreach (['p', 'street', 'city', 'postal'] as $v) {
				$data['pplPickup'][$k . '_' . $v] = $request->getPost('pplPickup')[$k . '_' . $v];
			}

			$data['disableDeliveryAddressSpedition'] = 'pplPickup';
			$data['disableDeliveryAddress']          = true;
			$data['useAddrDeli']                     = false;

			foreach (['street2', 'city2', 'postal2'] as $col) {
				if (isset($post[$col])) {
					$data[$col] = $post[$col];
				}
			}
		} else if ($data['disableDeliveryAddressSpedition'] === 'pplPickup') {
			unset($data['disableDeliveryAddress']);
			unset($data['useAddrDeli']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		$spedition = $this->getSpeditions()[$data['speditionCountry']][$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $v) {
				$d[$v . '2'] = $data[$v];
			}
			$countryId   = $data['speditionCountry'];
			$countryText = $this->countries->getAllNameColumn()[$countryId] ?? $countryId;
			$d           += [
				'useAddrDeli'  => true,
				'street2'      => $data['pplPickup'][$k . '_street'],
				'city2'        => $data['pplPickup'][$k . '_city'],
				'postal2'      => $data['pplPickup'][$k . '_postal'],
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form   = $event->form;
		$values = $form->getValues();

		$spedition = $this->getSpeditions()[$values->speditionCountry][$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition && $spedition->getSpedition()->isPickup) {
			$branchId = $values->pplPickup->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$branchId) {
				$form->addError($this->translator->translate('pplFront.orderForm.pplIdMissing'));
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		$spedition = $this->getSpeditions()[$values->speditionCountry][$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition) {
			$entity = new PplOrder($event->order);

			if ($spedition->getSpedition()->isPickup) {
				$k               = $values->speditionCountry . '_' . $spedition->getSpedition()->getId();
				$entity->pointId = (string) $values->pplPickup->$k;

				$orderSpedition = $event->order->getSpedition();
				if ($orderSpedition) {
					$orderSpedition->deliveryPointId = (string) $values->pplPickup->$k;
					$this->em->persist($orderSpedition);
				}
			}

			$pplProduct = explode(';', (string) $spedition->getSpedition()->code1);
			if (isset($pplProduct[0])) {
				$entity->pplProduct = (int) $pplProduct[0];
			}

			if (isset($pplProduct[1])) {
				$entity->pplProductCod = (int) $pplProduct[1];
			}

			$this->em->persist($entity);
		}
	}

	/**
	 * @return PaymentSpedition[][]
	 */
	protected function getSpeditions(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() !== 'ppl') {
				continue;
			}

			foreach ($row->getCountries() as $country) {
				$result[$country->getId()][$row->getPayment()->getId() . '_' . $row->getSpedition()->getId()] = $row;
			}
		}

		return $result;
	}

	protected function getPplSpeditionIds(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() !== 'ppl') {
				continue;
			}

			$result[$row->getSpedition()->getId()] = $row;
		}

		return $result;
	}

}
