<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Author;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use RealEstates\AdminModule\Model\Authors;
use RealEstates\Model\Entities\Author;

/**
 * Class AuthorForm
 * @package RealEstates\AdminModule\Components
 */
class AuthorForm extends BaseControl
{
	/** @var int @persistent */
	public $authorId;

	/** @var Author */
	public $author;

	/** @var Authors */
	protected $authorsService;

	public function __construct(Authors $authors)
	{
		$this->authorsService = $authors;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->authorId && !$this->author) {
			$this->setAuthor($this->authorId);
		}
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('name', 'default.name')->setRequired()->setMaxLength(255);
		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		if (($this->author && strtolower($this->author->name) != strtolower($values->name) || !$this->author) && $this->authorsService->getByName($values->name)) {
			$form->addError('realEstates.authorForm.authorExists');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->author) {
				$author       = $this->author;
				$flashMessage = 'realEstates.authorForm.edited';
			} else {
				$author       = new Author($values->name);
				$flashMessage = 'realEstates.authorForm.added';
			}

			$author->name = $values->name;

			$this->em->persist($author)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setAuthor($id)
	{
		$this->author   = $this->authorsService->get($id);
		$this->authorId = $id;

		if ($this->author) {
			$this['form']->setDefaults([
				'name' => $this->author->name,
			]);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/**
	 * @return Author
	 */
	public function getAuthor() { return $this->author; }
}