<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Variant;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use RealEstates\AdminModule\Model\Params;
use RealEstates\AdminModule\Model\Variants;
use RealEstates\Model\Entities\Param;
use RealEstates\Model\Entities\Variant;
use RealEstates\Model\Entities\VariantParam;

class VariantForm extends BaseControl
{
	/** @var Variant */
	public $variant;

	/** @var Param[] */
	protected $allParams;

	/** @var Variants */
	protected $variantsService;

	/** @var Params */
	protected $paramsService;

	public function __construct(Variants $variants, Params $params)
	{
		$this->variantsService = $variants;
		$this->paramsService   = $params;
	}

	public function render()
	{
		$this->template->allParams = $this->getAllParams();
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);

		$form->addComponent(new BaseContainer(), 'params');
		foreach ($this->getAllParams() as $param) {
			$form['params']->addComponent(new BaseContainer(), 'p_' . $param->getId());
			$form['params']['p_' . $param->getId()]->addCheckBox('allow', 'realEstates.variantForm.allow');
			$form['params']['p_' . $param->getId()]->addText('pattern', 'realEstate.variantForm.pattern');
		}

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			if ($this->variant) {
				$variant      = $this->variant;
				$flashMessage = 'realEstates.variantForm.edited';
			} else {
				$variant      = new Variant($values->title);
				$flashMessage = 'realEstates.variantForm.added';
			}

			$variant->title       = $values->title;
			$variant->isPublished = $values->isPublished;
			$this->em->persist($variant);

			$params        = [];
			$currentParams = [];

			// Pole parametrů entity podle ID
			foreach ($variant->getParams() as $p) {
				$currentParams[$p->param->getId()] = $p;
			}

			// Pole parametrů formuláře podle ID
			foreach ($values->params as $k => $v) {
				if (!$v->allow)
					continue;
				$k          = (int) ltrim($k, 'p_');
				$params[$k] = $v;
			}

			// Vytvoření a upravení parametrů z formuláře
			foreach ($params as $paramId => $data) {
				$paramEntity = $this->paramsService->getReference($paramId);
				$vp          = $currentParams[$paramId] ?? new VariantParam($variant, $paramEntity);

				$vp->setPattern($data->pattern);
				$this->em->persist($vp);
				$variant->addParam($vp);
			}

			// Odstranění ostatních
			foreach (array_diff_key($currentParams, $params) as $r) {
				$this->em->remove($r);
			}

			$this->em->flush();
			$this->em->commit();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setVariant($id)
	{
		$this->variant = $this->variantsService->get($id);

		if ($this->variant) {
			$this['form']->setDefaults([
				'title'       => $this->variant->title,
				'isPublished' => $this->variant->isPublished,
			]);


			/** @var VariantParam $param */
			foreach ($this->variant->getParams()->toArray() as $param) {
				$this['form']['params']['p_' . $param->param->getId()]['allow']->setDefaultValue(1);
				$this['form']['params']['p_' . $param->param->getId()]['pattern']->setDefaultValue($param->getPatternRaw());
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	public function getAllParams()
	{
		if (!$this->allParams) {
			$this->allParams = $this->paramsService->getEr()->createQueryBuilder('p')->orderBy('p.position', 'ASC')->getQuery()->getResult();
		}

		return $this->allParams;
	}
}
