<?php declare(strict_types = 1);

namespace RealEstates\FrontModule\Model\Dao;

use Core\Model\Providers\ISiteMapUrl;
use Gallery\FrontModule\Model\Dao\Album;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class Property
{
	use MagicAccessors;

	/** @var int */
	public $id;

	/** @var string */
	public $title;

	/** @var string */
	public $alias;

	/** @var string */
	public $description;

	/** @var Type */
	protected $type;

	/** @var string */
	public $image;

	/** @var string */
	public $rooms;

	/** @var int */
	public $roomsId;

	/** @var string */
	public $author;

	/** @var array */
	public $nextAuthors;

	/** @var float */
	public $usableArea;

	/** @var float */
	public $groundFloorArea;

	/** @var float */
	public $basePlateGroundFloor;

	/** @var int */
	public $basePlateGroundFloorPrice;

	/** @var float */
	public $garretArea;

	/** @var float */
	public $garageArea;

	/** @var float */
	public $basePlateGarage;

	/** @var int */
	public $basePlateGaragePrice;

	/** @var float */
	public $buildArea;

	/** @var int */
	public $basePlatePrice;

	/** @var float */
	public $vatModifier = 1;

	/** @var Album */
	protected $album;

	/** @var Variant[] */
	protected $variants;

	/** @var boolean */
	protected $modifyAvailable;

	/** @var array */
	protected $attribs;

	/** @var Filter[] */
	protected $filters;

	/** @var array */
	protected $seo = [];

	/** @var DateTime|null */
	protected $modified;

	/** @var int */
	protected $position;

	/** @var int */
	protected $views;

	/** @var bool */
	protected $isPublished;

	/** @var bool */
	protected $isOffline;

	/**
	 * @param Album $album
	 *
	 * @return $this
	 */
	public function setAlbum(Album $album)
	{
		$this->album = $album;

		return $this;
	}

	/**
	 * @param Variant $variant
	 *
	 * @return $this
	 */
	public function addVariant(Variant $variant)
	{
		$this->variants[] = $variant;

		return $this;
	}

	/**
	 * @param Filter $filter
	 *
	 * @return $this
	 */
	public function addFilter(Filter $filter)
	{
		$this->filters[$filter->id] = $filter;

		return $this;
	}

	public function getBasePrice()
	{
		$lowest = 99999999;
		foreach ($this->variants as $variant) {
			$p = $variant->getPrice();
			if ($p < $lowest && $p > 0) {
				$lowest = $variant->getPrice();
			}
		}

		return (int) ceil($lowest + $this->basePlatePrice);
	}

	public function getBasePriceVat()
	{
		return (int) ceil($this->getBasePrice() * $this->vatModifier);
	}

	public function getBasePlatePriceVat()
	{
		return (int) ceil($this->basePlatePrice * $this->vatModifier);
	}

	public function setType(Type $type)
	{
		$this->type = $type;

		return $this;
	}

	public function getSeo($key = null)
	{
		$tmp = $this->seo;
		if (!isset($tmp['title']))
			$tmp['title'] = $this->title . ' $separator $siteName';
		if (!isset($tmp['description']))
			$tmp['description'] = Strings::truncate(strip_tags($this->description), 255);

		return $key ? ($tmp[$key] ?? null) : $tmp;
	}

	/**
	 * @return DateTime|null
	 */
	public function getModified() { return $this->modified; }

	/**
	 * @return Album
	 */
	public function getAlbum() { return $this->album; }
}
