<?php declare(strict_types = 1);

namespace RealEstates\FrontModule\Model;

use Blog\Model\Authors;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Images;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use RealEstates\AdminModule\Model\Variants;
use RealEstates\Model\Entities\Author;
use RealEstates\Model\Entities\Filter;
use RealEstates\Model\Entities\Param;
use RealEstates\Model\Entities\Property;
use RealEstates\FrontModule\Model\Dao;
use Gallery\FrontModule\Model\Dao as DaoGallery;
use RealEstates\Model\Entities\PropertyParam;
use RealEstates\Model\Entities\Variant;
use RealEstates\Model\Helpers\Calculator;
use RealEstates\Model\Settings;

/**
 * TODO facade
 *
 * Class Propertys
 * @package RealEstates\FrontModule\Model
 */
class Propertys extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'realEstatesPropertys';

	protected $entityClass = Property::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var Albums */
	protected $albumsService;

	/** @var Images */
	protected $imagesService;

	/** @var Variants */
	protected $variantsService;

	/** @var Calculator */
	protected $calculator;

	/** @var Settings */
	protected $settingsService;

	/** @var PropertyTypes */
	protected $propertyTypesService;

	/** @var Rooms */
	protected $roomsService;

	/** @var Filters */
	protected $filtersService;

	/** @var Params */
	protected $paramsService;

	/** @var Dao\Property[][]|null */
	protected $publishedCached;

	/** @var array */
	protected $cViews;

	/** @var array */
	protected $filtersCached;

	/** @var array[][] */
	protected $filterEntities;

	/** @var bool */
	public $ignorePublish = false;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function __construct(Albums $albums, Images $images, Calculator $calculator, Settings $settings, DefaultLang $defaultLang,
	                            Variants $variants, PropertyTypes $propertyTypes, Rooms $rooms, Filters $filters, Params $params)
	{
		$this->albumsService        = $albums;
		$this->imagesService        = $images;
		$this->variantsService      = $variants;
		$this->settingsService      = $settings;
		$this->defaultLang          = $defaultLang;
		$this->propertyTypesService = $propertyTypes;
		$this->roomsService         = $rooms;
		$this->filtersService       = $filters;
		$this->paramsService        = $params;
		$this->calculator           = $calculator;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @return Criteria
	 * @throws \Exception
	 */
	protected function publishedCriteria()
	{
		$expr     = Criteria::expr();
		$now      = new DateTime();
		$locale   = $this->defaultLang->locale;
		$criteria = Criteria::create();

		if ($this->ignorePublish)
			return $criteria;

		$criteria->andWhere(
			$expr->andX(
				$expr->andX(
					$expr->orX($expr->isNull('p.publishUp'), $expr->lte('p.publishUp', $now)),
					$expr->orX($expr->isNull('p.publishDown'), $expr->gte('p.publishDown', $now))
				),
				$expr->neq('p.publishUp', null),
				$expr->eq('p.isPublished', 1),
				$expr->eq('p.isOffline', 0)
			//					,
			//					$expr->orX(
			//						$expr->isNull('c.lang'),
			//						$expr->eq('c.lang', $locale),
			//						$expr->eq('c.lang', '')
			//					)
			)
		);

		return $criteria;
	}

	/**
	 * @param $id
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		foreach ($this->getPublished() as $p) {
			if ($p->id == $id)
				return ['id' => $id, 'alias' => $p->alias];
		}

		$qb = $this->getEr()->createQueryBuilder('p')->select('p.id, p.alias')
			->andWhere(is_array($id) ? 'p.id IN (:id)' : 'p.id = :id')->setParameter('id', $id)
			->getQuery()->useResultCache(true, 60);

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	/**
	 * @param $propertyId
	 *
	 * @return bool
	 */
	public function addView($propertyId)
	{
		try {
			$this->em->createQuery('UPDATE ' . Property::class . ' p SET p.views = p.views + 1 WHERE p.id = :id')
				->setParameters([':id' => $propertyId])->execute();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function addPdfDownload($propertyId)
	{
		try {
			$this->em->createQuery('UPDATE ' . Property::class . ' p SET p.pdfDownloads = p.pdfDownloads + 1 WHERE p.id = :id')
				->setParameters([':id' => $propertyId])->execute();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/**
	 * @param null     $id
	 * @param string   $orderBy
	 * @param int|null $limit
	 * @param bool     $allowUnpublished
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getPublished($id = null, $orderBy = 'p.position ASC', int $limit = null, bool $allowUnpublished = false)
	{
		$key = implode('_', [(is_array($id) ? implode('-', $id) : $id), ($allowUnpublished ? 'au' : '')]);

		if (!$orderBy)
			$orderBy = 'p.position ASC';


		$data = $this->getCache()->load($key, function(&$dep) use ($id, $key, $allowUnpublished) {
			$dep = $this->cacheDep;

			if (is_numeric($id)) {
				foreach ($this->publishedCached as $c) {
					if (isset($c[$id])) {
						return $c[$id];
					}
				}
			}

			$propertys = [];
			$params    = [];

			//TODO partial select možnost optimalizace
			$propertysRaw = $this->getEr()->createQueryBuilder('p', 'p.id');

			if (!$allowUnpublished)
				$propertysRaw->addCriteria($this->publishedCriteria());

			$propertysRaw->addSelect('t, r, a, gallery')
				->join('p.type', 't')->join('p.rooms', 'r')->join('p.author', 'a')
				->leftJoin('p.gallery', 'gallery')
				->groupBy('p.id');

			if ($id) {
				$propertysRaw = $propertysRaw->andWhere(is_array($id) ? 'p.id IN (:id)' : 'p.id = :id')->setParameter('id', $id);
			}

			/** @var Property[] $propertysRaw */
			$propertysRaw   = $propertysRaw->getQuery()->getArrayResult();
			$propertysCount = count($propertysRaw);
			$propertysId    = array_keys($propertysRaw);

			if (!$propertysRaw)
				return [];

			/** @var PropertyParam[] $paramsRaw */
			$paramsRaw = $this->paramsService->getErPropertyParam()->createQueryBuilder('p', 'p.id');
			if ($id)
				$paramsRaw->andWhere('p.property = :id')->setParameter('id', $propertysId);
			$paramsRaw = $paramsRaw->getQuery()->getResult();
			foreach ($paramsRaw as $p) {
				if (isset($propertysRaw[$p->property->getId()]))
					$propertysRaw[$p->property->getId()]['params'][] = $p;
			}

			/** @var Filter[] $filtersRaw */
			$filtersRaw = $this->filtersService->getEr()->createQueryBuilder('f', 'f.id')->join('f.propertys', 'p')->addSelect('p as HIDDEN ps');
			if ($id)
				$filtersRaw->where('p.id = :id')->setParameter('id', $propertysId);
			$filtersRaw = $filtersRaw->getQuery()->getResult();
			foreach ($filtersRaw as $f) {
				foreach ($f->getPropertys() as $p) {
					if (isset($propertysRaw[$p->getId()]))
						$propertysRaw[$p->getId()]['filters'][$f->getId()] = $f;
				}
			}

			$albumsId = [];
			foreach ($propertysRaw as $p) {
				if ($p['gallery'])
					$albumsId[$p['id']] = $p['gallery']['id'];
			}

			/** @var Variant[] $variantsRaw */
			$variantsRaw = $this->variantsService->getEr()->createQueryBuilder('v', 'v . id')->where('v . isPublished = 1')->addSelect('vp, p')
				->join('v . params', 'vp')->join('vp . param', 'p')->getQuery()->getResult();


			$propertysRaw = ArrayHash::from($propertysRaw);

			$data = [
				'propertys' => $this->fillDao($propertysRaw, $variantsRaw),
				'albums'    => $albumsId,
			];

			if ($id && is_numeric($id))
				$this->publishedCached[$key][$id] = $data;
			else
				$this->publishedCached[$key] = $data;

			return is_numeric($id) ? ($this->publishedCached[$key][$id] ?: null) : $this->publishedCached[$key];
		});

		$propertys = [];
		foreach ($data['propertys'] as $k => $p) {
			if (($p->isOffline || !$p->isPublished) && !$allowUnpublished)
				continue;

			$propertys[$k] = $data['propertys'][$k];
			if (isset($data['albums'][$k]))
				$propertys[$k]->setAlbum($this->albumsService->get($data['albums'][$k]));

			if ($this->getViews()[$k]) {
				$propertys[$k]->setViews($this->getViews()[$k]);
			}
		}
		unset($data);

		$orderBy = array_pop(explode('p.', $orderBy));
		$orderBy = explode(' ', $orderBy, 2);

		if (!is_numeric($id)) {
			uasort($propertys, function($a, $b) use ($orderBy) {
				if (is_numeric($a->{$orderBy[0]}))
					return $a->{$orderBy[0]} <=> $b->{$orderBy[0]};
				else
					return strcmp($a->{$orderBy[0]}, $b->{$orderBy[0]});
			});

			if ($orderBy[1] == 'DESC')
				$propertys = array_reverse($propertys, true);

			if ($limit)
				$propertys = array_slice($propertys, 0, $limit, true);
		}

		return is_numeric($id) ? $propertys[$id] : $propertys;
	}

	public function getViews()
	{
		if (!$this->cViews)
			foreach ($this->getEr()->createQueryBuilder('p')->addCriteria($this->publishedCriteria())
				         ->select('p.id, p.views')->getQuery()->getArrayResult() as $v)
				$this->cViews[$v['id']] = $v['views'];

		return $this->cViews;
	}

	/**
	 * @param $alias
	 *
	 * @return Dao\Property|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getPublishedByAlias($alias)
	{
		foreach ($this->getPublished() as $p)
			if ($p->alias == $alias)
				return $p;

		return null;
	}

	public function getAllFilters($propertys)
	{
		$key = implode('', array_keys($propertys));

		if (isset($this->filtersCached[$key]))
			return $this->filtersCached[$key];

		$filters = [
			'type'            => [],
			'rooms'           => [],
			'price'           => [],
			'more'            => [],
			'usableArea'      => [],
			'usableAreaSteps' => [],
		];

		$prices      = [];
		$maxPrice    = 0;
		$lowestPrice = 99999999;
		$types       = [];

		$typeAll = $this->propertyTypesService->getPublished(1);
		if ($typeAll)
			$types[$typeAll['position']] = ArrayHash::from($typeAll);

		/** @var Dao\Property[] $propertys */
		foreach ($propertys as $property) {
			$price = $property->getBasePriceVat();

			$filters['rooms'][$property->roomsId]                       = $property->rooms;
			$filters['usableAreaSteps'][(string) $property->usableArea] = (float) $property->usableArea;
			$types[$property->type->position]                           = $property->type;

			$prices[] = $price;
			if ($price > $maxPrice)
				$maxPrice = $price;
			if ($price < $lowestPrice)
				$lowestPrice = $price;

			foreach ($property->filters as $f) {
				$filters['more'][$f->id] = $f->value;
			}
		}

		if ($lowestPrice <= 1500000)
			$filters['price']['lte1500000'] = $this->filterValueToText('price', 'lte1500000');

		$step = 500000;
		for ($i = 1500000; $i <= 3000000; $i += $step) {
			$top = $i + $step;
			foreach ($prices as $price) {
				if ($price >= $i && $price <= $top)
					$filters['price']['lte' . $top] = $this->filterValueToText('price', 'lte' . $top);
			}
		}

		if ($maxPrice > 3000000)
			$filters['price']['gte3000000'] = $this->filterValueToText('price', 'gte3000000');
		$filters['price'] = array_reverse($filters['price'], true);

		asort($filters['rooms'], SORT_STRING);
		$filters['rooms'] = array_reverse($filters['rooms'], true);

		ksort($filters['usableAreaSteps'], SORT_NUMERIC);

		$usableAreaDefaultKey = min(array_keys($filters['usableAreaSteps'])) . ' - ' . max(array_keys($filters['usableAreaSteps']));
		foreach ($filters['usableAreaSteps'] as $a => $ta)
			foreach ($filters['usableAreaSteps'] as $b => $tb) {
				if ($b < $a)
					continue;
				$filters['usableArea'][$a . ' - ' . $b] = $a . ' - ' . $b == $usableAreaDefaultKey ? '' : $this->filterValueToText('usableArea', $a . ' - ' . $b);
			}

		ksort($types);
		foreach ($types as $type)
			$filters['type'][$type->id] = $type->title;

		return $this->filtersCached[$key] = $filters;
	}

	public function filterValueToText($key, $value, $implodeArray = ' - ')
	{
		switch ($key) {
			case 'type':
				$type = $this->getFilterEntities('type', $value);
				if ($type)
					return $type['title'];

				return '';
			case 'rooms':
				$value = is_array($value) ? $value : explode(' | ', (string) $value);
				$rooms = count($value) > 1
					? ['value' => $this->t('front.realEstates.filter.inMoreRooms')] : $this->getFilterEntities('rooms', $value[0]);
				if ($rooms)
					return $rooms['value'];

				return '';
			case 'price':
				if (strpos($value, 'lte') === 0) {
					$key = 'priceLte';
					if (substr($value, 3) < 2000000)
						$key .= '1';
				} else if (strpos($value, 'gte') === 0) {
					$key = 'priceGte';
				} else {
					return $value;
				}

				return $this->t('front.realEstates.filter.' . $key, ['price' => str_replace(' . ', ',', ((int) substr($value, 3)) / 1000000)]);
			case 'more':
				$more = is_array($value) ? $value : explode(' | ', (string) $value);
				$tmp  = [];
				foreach ($more as $m)
					$tmp[] = $this->getFilterEntities('more', $m)['value'];
				if ($tmp)
					return $implodeArray ? implode($implodeArray, $tmp) : $tmp;

				return '';
			case 'usableArea':
				return $this->getFilterEntities('usableArea', $value);
		}

		return '';
	}

	public function filterTextToValue($key, $text)
	{
		$filters = $this->getAllFilters($this->getPublished());
		foreach ($filters[$key] as $k => $v) {
			if ($v == $text)
				return $k;
		}

		return $text;
	}

	/**
	 * @param Dao\Property[] $propertys
	 * @param array          $filters
	 *
	 * @return Dao\Property[]
	 */
	public function proceedFilter($propertys, $filters)
	{
		$filtered = [];

		foreach ($propertys as $property) {
			$match = [
				'type'       => !$filters['type'] || $property->type->id == $filters['type'] ? true : false,
				'rooms'      => !$filters['rooms'] || in_array($property->roomsId, $filters['rooms']) ? true : false,
				'price'      => $filters['price'] ? false : true,
				'more'       => $filters['more'] ? false : true,
				'usableArea' => $filters['usableArea'] ? false : true,
			];

			if (!$match['price']) {
				$price = $property->getBasePriceVat();

				$filterPrice = (int) substr($filters['price'], 3);
				if (strpos($filters['price'], 'lte') === 0 && $price <= $filterPrice) {
					$match['price'] = true;
				} else if (strpos($filters['price'], 'gte') === 0 && $price > $filterPrice) {
					$match['price'] = true;
				}
			}

			if (!$match['more']) {
				if ($filters['more'] && !$property->filters)
					$match['more'] = false;
				else {
					$match['more'] = true;

					foreach ($filters['more'] as $more)
						if (!array_key_exists($more, $property->filters)) {
							$match['more'] = false;
							break;
						}
				}
			}

			if (!$match['usableArea']) {
				$area = explode(' - ', $filters['usableArea']);
				if ($property->usableArea >= $area[0] && $property->usableArea <= $area[1])
					$match['usableArea'] = true;
			}

			if (!in_array(false, $match))
				$filtered[] = $property;
		}

		return $filtered;
	}

	protected function fillDao($propertysRaw, $variantsRaw = null)
	{
		foreach ($propertysRaw as $pr) {
			/** @var Property $pr */
			$property = new Dao\Property();
			$property
				->setId($pr->id)
				->setTitle($pr->title)
				->setAlias($pr->alias)
				->setDescription($pr->text)
				->setRooms($pr->rooms->value)->setRoomsId($pr->rooms->id)
				->setAuthor($pr->author->name)
				->setUsableArea($pr->groundFloorArea + $pr->garretArea)
				->setBuildArea($this->calculator->basePlateSize($pr->groundFloorArea + $pr->garageArea))
				->setGroundFloorArea($pr->groundFloorArea)
				->setGarretArea($pr->garretArea)
				->setGarageArea($pr->garageArea)
				->setBasePlateGroundFloor($this->calculator->basePlateSize($pr->groundFloorArea))
				->setBasePlateGroundFloorPrice($this->calculator->basePlateGroundFloorPrice($pr->groundFloorArea))
				->setBasePlateGarage($this->calculator->basePlateSize($pr->garageArea))
				->setBasePlateGaragePrice($this->calculator->basePlateGaragePrice($pr->garageArea))
				->setBasePlatePrice($property->basePlateGroundFloorPrice + $property->basePlateGaragePrice)
				->setModifyAvailable($pr->modifyAvailable)
				->setVatModifier(($this->settingsService->get('vat', 100) / 100) + 1)
				->setAttribs($pr->attribs)
				->setModified($pr->modified)
				->setPosition($pr->position)
				->setViews($pr->views)
				->setIsOffline($pr->isOffline)
				->setIsPublished($pr->isPublished)
				->setSeo($pr->seo);

			foreach ($pr->attribs['nextAuthors'] as $v)
				$property->nextAuthors[] = $v;

			$propertys[$pr->id] = $property;

			/** @var Param[] $params */
			$params = [];
			foreach ($pr->params as $prP) {
				$params[$prP->param->id] = $prP;
			}

			// Typ
			$type = (new Dao\Type())
				->setId($pr->type->id)
				->setTitle($pr->type->title)
				->setSingularTitle($pr->type->singularTitle)
				->setImage($pr->type->image)
				->setPosition($pr->type->position)
				->setParams($pr->type->params)
				->setSeoTitle($pr->type->seoTitle)
				->setSeoDescription($pr->type->seoDescription);
			$property->setType($type);

			// Filtr
			foreach ($pr->filters as $f) {
				$filter = (new Dao\Filter())
					->setId($f->id)
					->setValue($f->value);
				$property->addFilter($filter);
			}

			// Nastavení variant
			if ($variantsRaw) {
				foreach ($variantsRaw as $v) {
					$variant = (new Dao\Variant())
						->setId($v->getId())
						->setTitle($v->title)
						->setProperty($property);

					$tmp      = [];
					$tmpPrice = 0;

					foreach ($v->params as $vp) {
						$p = $vp->param;
						if (!isset($params[$p->getId()]) || $params[$p->getId()]->value <= 0)
							continue;

						$pp     = $params[$p->getId()];
						$result = $this->calculator->paramResult($vp->getPattern(), $vp->getPricePerUnit(), $pp->value);

						$tmp[$p->position] = (new Dao\Param())
							->setId($p->getId())
							->setTitle($p->title)
							->setDescription((is_array($pp->description) && isset($pp->description[$v->getId()]) ? $pp->description[$v->getId()] : $p->description))
							->setDescription2($p->description2)
							->setPricePerUnit($p->pricePerUnit)
							->setUnit($p->unit)
							->setValue($pp->value)
							->setResult($result);

						$tmpPrice += $result;
					}

					if (isset($pr->attribs->additionalParams)) {
						$adPPos = 10000;
						foreach ($pr->attribs->additionalParams as $adP) {
							if (!isset($adP['activeVariant'][$v->getId()]) || $adP['activeVariant'][$v->getId()] != 1)
								continue;

							$result       = intval($adP['value']) * intval($adP['pricePerUnit']);
							$tmp[$adPPos] = (new Dao\Param())
								->setTitle($adP['name'])
								->setDescription($adP['desc'][$v->getId()] ?? '')
								->setPricePerUnit($adP['pricePerUnit'])
								->setUnit($adP['unit'])
								->setValue($adP['value'])
								->setResult($result);

							$adPPos++;
							$tmpPrice += $result;
						}
					}

					ksort($tmp);
					$variant->setParams($tmp)
						->setPrice($tmpPrice)
						->setPriceVat($this->calculator->vatPrice($tmpPrice))
						->setVat($this->settingsService->get('vat'));

					$property->addVariant($variant);
				}
			}
		}

		return $propertys;
	}

	protected function getFilterEntities($type, $id = null)
	{
		if ($type == 'type' && !isset($this->filterEntities[$type])) {
			$this->filterEntities[$type] = $this->propertyTypesService->getPublished();
		} else if ($type == 'rooms' && !isset($this->filterEntities[$type])) {
			$this->filterEntities[$type] = $this->roomsService->getPublished();
		} else if ($type == 'more' && !isset($this->filterEntities[$type])) {
			$this->filterEntities[$type] = $this->filtersService->getPublished();
			//			$this->filterEntities[$type]['withGarage'] = ['id'    => 'withGarage',
			//			                                              'value' => $this->t('front.realEstates.filter.withGarage')];
		} else if ($type == 'usableArea' && !isset($this->filterEntities[$type][$id])) {
			if (!$id)
				$text = '';
			else {
				list($x, $y) = explode('-', $id);
				$text = $this->t('front.realEstates.filter.withUsableArea');
				$x    = trim($x);
				$y    = trim($y);
				if ($x == $y)
					$text .= ' ' . $this->t('front.realEstates.filter.area', ['area' => $x]);
				else
					$text .= ' ' . $this->t('front.realEstates.filter.fromArea', ['area' => $x]) . ' ' . $this->t('front.realEstates.filter.toArea', ['area' => $y]);
			}
			$this->filterEntities[$type][$id] = $text;
		}

		return $id ? $this->filterEntities[$type][$id] : $this->filterEntities[$type];
	}
}
