<?php declare(strict_types = 1);

namespace RealEstates\Model;

use Core\Model\Pdf\Pdf;
use Mpdf\Output\Destination;
use Nette\Application\BadRequestException;
use Nette\FileNotFoundException;
use Nette\Utils\FileSystem;
use RealEstates\FrontModule\Model\Propertys;

class PropertysPdf
{
	/** @var Pdf */
	protected $pdfService;

	/** @var Propertys */
	protected $propertysService;

	/** @var Settings */
	protected $settingsService;

	protected $templatesDir = TEMPLATES_DIR . '/Front/default/RealEstates/PdfTemplates';

	public function __construct(Pdf $pdf, Propertys $propertys, Settings $settings)
	{
		$this->pdfService       = $pdf;
		$this->propertysService = $propertys;
		$this->settingsService  = $settings;
	}

	public function generateCalculation($id, $regenerate = false, $allowUnpublished = false)
	{
		$tmpFile      = TMP_DIR . '/realEstates/propertysPdf/calculation_' . $id . '.pdf';
		$templateFile = $this->templatesDir . '/calculation.latte';

		if (!file_exists($templateFile))
			throw new FileNotFoundException($templateFile);

		$this->propertysService->ignorePublish = true;

		$property = $this->propertysService->getPublished($id, 'p.position ASC', null, $allowUnpublished);
		if (!$property)
			return false;

		if (!file_exists($tmpFile) || $regenerate) {
			try {
				set_time_limit(360);
				$pdf             = $this->pdfService->newPdf();
				$templateFactory = $this->pdfService->getTemplateFactory();
				$templateFactory->setFile($templateFile);
				$templateFactory->property = $property;

				FileSystem::createDir(dirname($tmpFile));
				$pdf->WriteHTML($templateFactory->renderToString());
				$pdf->Output($tmpFile, Destination::FILE);
			} catch (\Exception $e) {
				unlink($tmpFile);
				throw $e;
			}
		}

		return $tmpFile;
	}

	public function generateCatalog($regenerate = false)
	{
		$tmpFile      = TMP_DIR . '/realEstates/propertysPdf/catalog.pdf';
		$templateFile = $this->templatesDir . '/catalog.latte';

		if (!file_exists($templateFile))
			throw new FileNotFoundException($templateFile);

		$propertys = $this->propertysService->getPublished();

		if (!file_exists($tmpFile) || $regenerate) {
			try {
				set_time_limit(360);
				$pdf             = $this->pdfService->newPdf();
				$templateFactory = $this->pdfService->getTemplateFactory();
				$templateFactory->setFile($templateFile);

//				usort($propertys, function($a, $b) {
//					return $b->usableArea <=> $a->usableArea;
//				});
//
				usort($propertys, function($a, $b) {
					return $a->type->title <=> $b->type->title;
				});

				foreach ($propertys as &$property) {
					$images = $property->album->getImages();
					foreach ($images as $k => $i) {
						if ($i->isCover || $property->album->cover->id == $i->id) {
							unset($images[$k]);
							$property->album->setImages($images);
							break;
						}
					}
				}

				$templateFactory->propertys = $propertys;

				FileSystem::createDir(dirname($tmpFile));
				$pdf->WriteHTML($templateFactory->renderToString());
				$pdf->Output($tmpFile, Destination::FILE);
			} catch (\Exception $e) {
				unlink($tmpFile);
				throw $e;
			}
		}

		return $tmpFile;
	}

	public function downloadCalculation($id)
	{
		$tmpFile = $this->generateCalculation($id);

		if ($tmpFile) {
			$this->propertysService->addPdfDownload($id);

			return $tmpFile;
		}

		return null;
	}

	public function downloadCatalog()
	{
		$tmpFile = $this->generateCatalog(true);
		$this->settingsService->addCatalogDownload();

		return $tmpFile;
	}
}
