<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Filter;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use RealEstates\AdminModule\Model\Filters;
use RealEstates\Model\Entities\Filter;

/**
 * Class FilterForm
 * @package RealEstates\AdminModule\Components
 */
class FilterForm extends BaseControl
{
	/** @var int @persistent */
	public $filterId;

	/** @var Filter */
	public $filter;

	/** @var Filters */
	protected $filtersService;

	public function __construct(Filters $filters)
	{
		$this->filtersService = $filters;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->filterId && !$this->filter) {
			$this->setFilter($this->filterId);
		}
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('value', 'default.value')->setRequired()->setMaxLength(255);
		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		if (($this->filter && strtolower($this->filter->value) != strtolower($values->value) || !$this->filter) && $this->filtersService->getByValue($values->value)) {
			$form->addError('realEstates.filterForm.filterExists');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->filter) {
				$filter       = $this->filter;
				$flashMessage = 'realEstates.filterForm.edited';
			} else {
				$filter       = new Filter($values->value);
				$flashMessage = 'realEstates.filterForm.added';
			}

			$filter->value = $values->value;

			$this->em->persist($filter)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setFilter($id)
	{
		$this->filter   = $this->filtersService->get($id);
		$this->filterId = $id;

		if ($this->filter) {
			$this['form']->setDefaults([
				'value' => $this->filter->value,
			]);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/**
	 * @return Filter
	 */
	public function getFilter() { return $this->filter; }
}
