<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Property;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Entities\Album;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use RealEstates\AdminModule\Model\Authors;
use RealEstates\AdminModule\Model\Filters;
use RealEstates\AdminModule\Model\Params;
use RealEstates\AdminModule\Model\Propertys;
use RealEstates\AdminModule\Model\PropertyTypes;
use RealEstates\AdminModule\Model\Rooms;
use RealEstates\AdminModule\Model\Variants;
use RealEstates\Model\Entities\Property;
use RealEstates\Model\Entities\Param;
use RealEstates\Model\Entities\PropertyParam;
use RealEstates\Model\Entities\Variant;
use RealEstates\Model\PropertysPdf;

class PropertyForm extends BaseControl
{
	/** @var Property */
	public $property;

	/** @var Param[] */
	protected $allParams;

	/** @var Variant[] */
	protected $allVariants;

	/** @var Propertys */
	protected $propertysService;

	/** @var Rooms */
	protected $roomsService;

	/** @var Authors */
	protected $authorsService;

	/** @var PropertyTypes */
	protected $propertyTypesService;

	/** @var Params */
	protected $paramsService;

	/** @var Variants */
	protected $variantsService;

	/** @var Filters */
	protected $filtersService;

	/** @var Request */
	protected $httpRequest;

	/** @var IImagesZoneFactory */
	protected $imagesZoneFactory;

	/** @var PropertysPdf */
	protected $propertysPdf;

	/** @var SeoContainer */
	protected $seoContainerService;

	/** @var array */
	protected $dynamicTabs;

	/** @var callable */
	public $onEditBeforeSave = [];

	public function __construct(Propertys $propertys, PropertyTypes $propertyTypes, Params $params, Request $request, IImagesZoneFactory $imagesZoneFactory,
	                            Rooms $rooms, Authors $authors, Variants $variants, Filters $filters, PropertysPdf $propertysPdf, SeoContainer $seoContainer)
	{
		$this->propertysService     = $propertys;
		$this->propertyTypesService = $propertyTypes;
		$this->paramsService        = $params;
		$this->httpRequest          = $request;
		$this->imagesZoneFactory    = $imagesZoneFactory;
		$this->roomsService         = $rooms;
		$this->authorsService       = $authors;
		$this->variantsService      = $variants;
		$this->filtersService       = $filters;
		$this->propertysPdf         = $propertysPdf;
		$this->dynamicTabs          = [];
		$this->seoContainerService  = $seoContainer;
	}

	public function render()
	{
		$this->template->property    = $this->property;
		$this->template->allParams   = $this->getAllParams();
		$this->template->allVariants = $this->getAllVariants();
		$this->template->dynamicTabs = $this->dynamicTabs;

		$variantsAllowedParams = [];
		$variantsParamsPattern = [];
		foreach ($this->getAllVariants() as $variant) {
			$variantsAllowedParams[$variant->getId()] = [];

			foreach ($variant->getParams() as $vp) {
				$variantsAllowedParams[$variant->getId()][$vp->param->getId()] = $vp;

				if ($vp->getPatternRaw()) {
					$variantsParamsPattern[$variant->getId() . '-' . $vp->param->getId()] = $this->paramsService->preparePattern($vp->getPatternRaw(), $vp->getPricePerUnit());
				}
			}
		}
		$this->template->variantsAllowedParams = $variantsAllowedParams;
		$this->template->variantsParamsPattern = $variantsParamsPattern;

		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handlePreSave()
	{
		$presenter = $this->getPresenter();
		$data      = $this->httpRequest->getPost('formData');
		$isOffline = $this->getPresenter()->getName() == 'RealEstates:Admin:Offline' ? true : false;
		if ($this->property || !$data)
			$presenter->sendPayload();

		$property              = new Property($data['title']);
		$property->publishUp   = new DateTime();
		$property->isOffline   = $isOffline ? 1 : 0;
		$property->isPublished = 0;
		$this->em->persist($property)->flush();
		$presenter->flashMessageSuccess('realEstates.propertyForm.added');
		$presenter->redirect(($isOffline ? 'Offline' : 'Online') . ':editProperty', ['id' => $property->getId()]);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$types = [];
		foreach ($this->propertyTypesService->getEr()->createQueryBuilder('pt')->select('pt.id, pt.title')->orderBy('pt.position', 'ASC')->getQuery()->getArrayResult() as $pt) {
			$types[$pt['id']] = $pt['title'];
		}

		$rooms = [];
		foreach ($this->roomsService->getEr()->createQueryBuilder('r')->select('r.id, r.value')->orderBy('r.value', 'ASC')->getQuery()->getArrayResult() as $r) {
			$rooms[$r['id']] = $r['value'];
		}

		$authors = [];
		foreach ($this->authorsService->getEr()->createQueryBuilder('a')->select('a.id, a.name')->orderBy('a.name', 'ASC')->getQuery()->getArrayResult() as $a) {
			$authors[$a['id']] = $a['name'];
		}

		$variants = [];
		foreach ($this->getAllVariants() as $v) {
			$variants[$v->getId()] = $v->title;
		}

		$filters = [];
		foreach ($this->filtersService->getEr()->createQueryBuilder('f')->select('f.id, f.value')->orderBy('f.value', 'ASC')->getQuery()->getArrayResult() as $f) {
			$filters[$f['id']] = $f['value'];
		}

		$form->addText('title', 'default.title')->setMaxLength(255)->setRequired();
		$form->addText('alias', 'default.alias')->setMaxLength(255);
		$form->addSelect('type', 'realEstates.propertyForm.type', $types)->setRequired();
		$form->addSelect('rooms', 'realEstates.propertyForm.rooms', $rooms)->setRequired();
		$form->addSelect('author', 'realEstates.propertyForm.author', $authors)->setRequired();
		$form->addTextArea('nextAuthors', 'realEstates.propertyForm.nextAuthors');
		$form->addText('groundFloorArea', 'realEstates.propertyForm.groundFloorArea')->setType('number')->setAttribute('step', .1);
		$form->addText('garretArea', 'realEstates.propertyForm.garretArea')->setType('number')->setAttribute('step', .1);
		$form->addText('garageArea', 'realEstates.propertyForm.garageArea')->setType('number')->setAttribute('step', .1);
		$form->addBool('modifyAvailable', 'realEstates.propertyForm.modifyAvailable')->setDefaultValue(1);
		$form->addCheckboxList('filters', 'realEstates.propertyForm.moreFilters', $filters);
		$form->addEditor('introtext', 'realEstates.propertyForm.introtext')->setToolbar('Text')->setDisableAutoP()->setHeight(100);
		$form->addEditor('text', 'default.text')->setDisableAutoP(false)->setToolbar('Property');
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(0);
		$form->addBool('isOffline', 'realEstates.propertyForm.isOffline');
		$form->addDateTimePicker('publishUp', 'default.publishUp')->setDefaultValue(new DateTime())->setRequired();
		$form->addDateTimePicker('publishDown', 'default.publishDown');
		$form->addText('views', 'realEstates.propertyForm.views')->setDisabled(true)->setDefaultValue(0);

		$form->addCheckbox('regenerateCalculation', 'realEstates.propertyForm.regenerateCalculation')->setDefaultValue(1);
		$form->addCheckbox('regenerateCatalog', 'realEstates.propertyForm.regenerateCatalog')->setDefaultValue(1);

		$form->onAnchor[] = function() use ($form) {
			if (!$this->property) {
				$form['isOffline']->setDefaultValue($this->getPresenter()->getName() == 'RealEstates:Admin:Offline' ? 1 : 0);
			}
		};

		$form->addComponent(new BaseContainer(), 'params');
		foreach ($this->getAllParams() as $param) {
			$form['params']->addComponent(new BaseContainer(), 'p_' . $param['id']);
			$form['params']['p_' . $param['id']]->addText('value')->setType('number')->setAttribute('step', .1);

			$form['params']['p_' . $param['id']]->addComponent(new BaseContainer(), 'desc');

			foreach ($variants as $k => $v) {
				$form['params']['p_' . $param['id']]['desc']->addText('d_' . $k)->setMaxLength(255);
			}
		}

		$form->addComponent(new BaseContainer(), 'additionalParams');
		$form->addComponent($this->seoContainerService->getContainer(), 'seo');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			if ($this->property) {
				$property     = $this->property;
				$flashMessage = 'realEstates.propertyForm.edited';
			} else {
				$property     = new Property($values->title);
				$flashMessage = 'realEstates.propertyForm.added';
			}

			$property->title = $values->title;
			$property->setAlias($values->alias);
			$property->setGroundFloorArea($values->groundFloorArea);
			$property->setGarretArea($values->garretArea);
			$property->setGarageArea($values->garageArea);
			$property->setSeo($this->seoContainerService->getFormData($values->seo));
			$property->type            = $this->propertyTypesService->getReference($values->type);
			$property->rooms           = $this->roomsService->getReference($values->rooms);
			$property->modifyAvailable = $values->modifyAvailable;
			$property->author          = $this->authorsService->getReference($values->author);
			$property->introtext       = $values->introtext;
			$property->text            = $values->text;
			$property->isPublished     = $values->isPublished;
			$property->isOffline       = $values->isOffline;
			$property->publishUp       = $values->publishUp;
			$property->publishDown     = $values->publishDown;

			$this->em->persist($property)->flush();

			// Dodatečné parametry
			$additionalParams = [];
			foreach ($this->getPresenter()->getHttpRequest()->getPost('additionalParams') as $row)
				if ($row['name'])
					$additionalParams[] = $row;
			unset($additionalParams['example']);
			$property->setAttr('additionalParams', $additionalParams);

			$params        = [];
			$currentParams = [];

			// Pole parametrů entity podle ID
			foreach ($property->getParams() as $p) {
				$currentParams[$p->param->getId()] = $p;
			}

			// Pole parametrů formuláře podle ID
			foreach ($values->params as $k => $v) {
				if ($v->value == '')
					continue;
				$k          = (int) ltrim($k, 'p_');
				$params[$k] = $v;
			}

			// Další filtry
			$f = [];
			foreach ($values->filters as $filterId)
				$f[$filterId] = $this->filtersService->getReference((int) $filterId);
			$property->setFilters($f);

			// Další autoří
			$property->setAttr('nextAuthors', $values->nextAuthors ? explode("\n", $values->nextAuthors) : []);

			// Vytvoření a upravení parametrů z formuláře
			foreach ($params as $paramId => $data) {
				$descs = [];

				foreach ($data->desc as $vId => $vDesc) {
					if ($vDesc) {
						$descs[ltrim($vId, 'd_')] = $vDesc;
					}
				}

				$paramEntity = $this->paramsService->getReference($paramId);
				$pp          = $currentParams[$paramId] ?? new PropertyParam($property, $paramEntity, $data->value);

				$pp->value = $data->value;
				$pp->setDescription($descs);
				$this->em->persist($pp);
				$property->addParam($pp);
			}

			// Odstranění ostatních
			foreach (array_diff_key($currentParams, $params) as $r) {
				$this->em->remove($r);
			}

			foreach ($this->onEditBeforeSave as $c) {
				$c($form, $values, $property);
			}

			$this->em->persist($property)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
			$form->addCustomData('propertyId', $property->getId());
			$this->em->commit();
			$this->property = $property;
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());

			return false;
		}

		if ($values->regenerateCalculation)
			$this->propertysPdf->generateCalculation($this->property->getId(), true);

		if ($values->regenerateCatalog)
			$this->propertysPdf->generateCatalog(true);
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			if (!$this->property->gallery) {
				$album        = new Album(UPLOADS_PATH . '/real_estates/property');
				$album->title = $this->property->title;
				$this->em->persist($album)->flush();

				$control->setAlbum($album->getId());
				$this->property->gallery = $album;

				$this->em->persist($this->property)->flush();
			}
		};

		$control->onAfterUpdate[] = function($image) {
			$this->propertysService->clearFrontCache();
		};

		if ($this->property && $this->property->gallery) {
			$control->setAlbum($this->property->gallery->getId());
		}

		return $control;
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setProperty($id)
	{
		$this->property = $this->propertysService->getWithJoins($id);

		if ($this->property) {
			$p = $this->property;
			$this['form']->setDefaults([
				'title' => $p->title,
				'alias' => $p->alias,
			]);

			$p = $this->property;
			$this['form']->setDefaults([
				'title'           => $p->title,
				'alias'           => $p->alias,
				'groundFloorArea' => str_replace(',', '.', $p->groundFloorArea),
				'garretArea'      => str_replace(',', '.', $p->garretArea),
				'garageArea'      => str_replace(',', '.', $p->garageArea),
				'modifyAvailable' => $p->modifyAvailable,
				'introtext'       => $p->introtext,
				'text'            => $p->text,
				'isPublished'     => $p->isPublished,
				'isOffline'       => $p->isOffline,
				'publishUp'       => $p->publishUp,
				'publishDown'     => $p->publishDown,
				'nextAuthors'     => $p->getAttr('nextAuthors') ? implode("\n", $p->getAttr('nextAuthors')) : null,
			]);

			if ($p->type && array_key_exists($p->type->getId(), $this['form']['type']->getItems())) {
				$this['form']['type']->setDefaultValue($p->type->getId());
			}

			if ($p->rooms && array_key_exists($p->rooms->getId(), $this['form']['rooms']->getItems())) {
				$this['form']['rooms']->setDefaultValue($p->rooms->getId());
			}

			if ($p->author && array_key_exists($p->author->getId(), $this['form']['author']->getItems())) {
				$this['form']['author']->setDefaultValue($p->author->getId());
			}

			$d = [];
			foreach ($p->getFilters() as $f) {
				if (array_key_exists($f->getId(), $this['form']['filters']->getItems()))
					$d[] = $f->getId();
				$this['form']['filters']->setDefaultValue($d);
			}

			/** @var PropertyParam $param */
			foreach ($p->getParams()->toArray() as $param) {
				$tmp = 'params-p_' . $param->param->getId();
				$this['form'][$tmp . '-value']->setDefaultValue(str_replace(',', '.', $param->value));

				if (is_array($param->getDescription())) {
					foreach ($param->getDescription() as $k => $v) {
						$tmp2 = $tmp . '-desc-d_' . $k;
						if (isset($this['form'][$tmp2]))
							$this['form'][$tmp2]->setDefaultValue($v);
					}
				}
			}

			$this->seoContainerService->setDefaults($this['form']['seo'], $p->getSeo());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getAllParams()
	{
		if (!$this->allParams) {
			$this->allParams = $this->paramsService->getEr()->createQueryBuilder('pp')->orderBy('pp.position', 'ASC')->getQuery()->getArrayResult();

			foreach ($this->allParams as &$param) {
				$param['pattern'] = $this->paramsService->preparePattern($param['pattern'], $param['pricePerUnit']);
			}
		}

		return $this->allParams;
	}

	protected function getAllVariants()
	{
		if (!$this->allVariants) {
			$this->allVariants = $this->variantsService->getEr()->createQueryBuilder('v')
				->join('v.params', 'vp')->join('vp.param', 'param')
				->addSelect('vp, param')
				->orderBy('v.position', 'ASC')->getQuery()->getResult();
		}

		return $this->allVariants;
	}
}
