<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Property;

use Core\Model\UI\BaseControl;
use Kdyby\Doctrine\QueryBuilder;
use Nette\Application\Responses\FileResponse;
use Nette\Utils\Html;
use RealEstates\AdminModule\Model\Authors;
use RealEstates\AdminModule\Model\Propertys;
use RealEstates\AdminModule\Model\PropertyTypes;
use RealEstates\AdminModule\Model\Rooms;
use RealEstates\Model\Entities\Property;
use RealEstates\Model\PropertysPdf;

class PropertysGrid extends BaseControl
{
	/** @var Propertys */
	protected $propertysService;

	/** @var Rooms */
	protected $roomsService;

	/** @var Authors */
	protected $authorsService;

	/** @var PropertyTypes */
	protected $propertyTypesService;

	/** @var PropertysPdf */
	protected $propertysPdf;

	/** @var QueryBuilder */
	protected $qb;

	/** @var bool */
	public $calculationRegenerate = false;

	public function __construct(Propertys $propertys, Rooms $rooms, Authors $authors, PropertyTypes $propertyTypes, PropertysPdf $propertysPdf)
	{
		$this->propertysService     = $propertys;
		$this->roomsService         = $rooms;
		$this->authorsService       = $authors;
		$this->propertyTypesService = $propertyTypes;
		$this->propertysPdf         = $propertysPdf;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->propertysService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handlePdf($id)
	{
		$property = $this->propertysService->get($id);
		$name = $this->translator->translate('front.realEstates.calculationTitle', null, ['name' => $property->title]);
		$this->getPresenter()->sendResponse(new FileResponse($this->propertysPdf->generateCalculation($id, $this->calculationRegenerate, true), $name, null, false));
	}

	public function handleResetPdfDownloads($id)
	{
		$presenter = $this->getPresenter();
		if ($this->propertysService->resetPdfDownloads($id)) {
			$presenter->flashMessageSuccess('realEstates.propertysGrid.pdfDownloadsReseted');
		} else {
			$presenter->flashMessageDanger('realEstates.propertysGrid.pdfDownloadsResetFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleResetViews($id)
	{
		$presenter = $this->getPresenter();
		if ($this->propertysService->resetViews($id)) {
			$presenter->flashMessageSuccess('realEstates.propertysGrid.viewsReseted');
		} else {
			$presenter->flashMessageDanger('realEstates.propertysGrid.viewsResetFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setSortable();
		$grid->setSortableHandler('propertysGrid:gridSortableRow!');

		$this->roomsService->getAll();
		$this->authorsService->getAll();

		$qb = $this->getQb();
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('title', 'default.title')->setRenderer(function($row) {
			$linkPresenter = $this->getPresenter()->getName() == 'RealEstates:Admin:Offline' ? 'Offline' : 'Online';

			return Html::el('a', ['href' => $this->getPresenter()->link($linkPresenter . ':editProperty', [$row->getId()])])->setText($row->title);
		});
		$grid->addColumnText('type', 'realEstates.propertysGrid.type', 'type.title');
		$grid->addColumnText('rooms', 'realEstates.propertysGrid.rooms', 'rooms.value');
		$grid->addColumnText('author', 'realEstates.propertysGrid.author', 'author.name');
		$grid->addColumnNumber('views', 'realEstates.propertysGrid.views')->setRenderer(function($row) {
			return Html::el()
				->addHtml(Html::el('span class="mr-2"')->setText($row->views . 'x'))
				->addHtml(Html::el('a class="btn btn-xs btn-danger ajax"', ['href' => $this->link('resetViews!', ['id' => $row->getId()])])->setText('x'));
		});
		$grid->addColumnNumber('pdfDownloads', 'realEstates.propertysGrid.pdfDownloads')->setRenderer(function($row) {
			return Html::el()
				->addHtml(Html::el('span class="mr-2"')->setText($row->pdfDownloads . 'x'))
				->addHtml(Html::el('a class="btn btn-xs btn-danger ajax"', ['href' => $this->link('resetPdfDownloads!', ['id' => $row->getId()])])->setText('x'));
		});
		$grid->addColumnText('area', 'realEstates.propertysGrid.usableArea')->setRenderer(function($row) {
			/** @var Property $row */
			return Html::el()->setHtml(number_format($row->getUsableArea(), 2, ',', ' ') . ' m<sup>2</sup>');
		})->setAlign('right');
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('title', '');

		$types = [];
		foreach ($this->propertyTypesService->getAll() as $type)
			$types[$type->getId() == 1 ? '' : $type->getId()] = $type->title;
		$grid->addFilterSelect('type', '', $types);

		$rooms = ['' => ''];
		foreach ($this->roomsService->getAll() as $room)
			$rooms[$room->getId()] = $room->value;
		$grid->addFilterSelect('rooms', '', $rooms);

		$authors = ['' => ''];
		foreach ($this->authorsService->getAll() as $author)
			$authors[$author->getId()] = $author->name;
		$grid->addFilterSelect('author', '', $authors);

		$grid->addFilterSelect('isPublished', '', ['' => '', 0 => $this->t('default.no'),
		                                           1  => $this->t('default.yes')]);

		// Actions
		$grid->addAction('pdf', 'pdf', '')->setRenderer(function($row) {
			return Html::el('a class="btn btn-xs btn-info"', [
				'href'   => $this->link('pdf!', $row->getId()),
				'target' => '_blank'])
				->addHtml(Html::el('i class="fa fa-file-pdf"'))
				->addHtml(Html::el('span class=ml-1')->setText($this->t('realEstates.propertysGrid.pdfCalculation')));
		});
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		$grid->setRowCallback(function($row, $tr) {
			/** @var Html $tr */
			$tr->addClass('group_' . ($row->isOffline ? 1 : 0));
			$tr->addAttributes(['data-group' => ($row->isOffline ? 1 : 0)]);

			return $tr;
		});

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->propertysService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->propertysService->getEr()->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Get / Set
	 */

	/**
	 * @return QueryBuilder
	 */
	public function getQb()
	{
		if (!$this->qb) {
			$this->qb = $this->propertysService->getEr()->createQueryBuilder('p')->orderBy('p.position', 'ASC')
				->join('p.type', 't')->join('p.rooms', 'r')->join('p.author', 'a')->addSelect('t, r, a');
		}

		return $this->qb;
	}

	/**
	 * @param QueryBuilder $qb
	 */
	public function setQb($qb) { $this->qb = $qb; }
}
