<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Components\Room;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use RealEstates\AdminModule\Model\Rooms;
use RealEstates\Model\Entities\Room;

class RoomForm extends BaseControl
{
	/** @var int @persistent */
	public $roomId;

	/** @var Room */
	public $room;

	/** @var Rooms */
	protected $roomsService;

	public function __construct(Rooms $rooms)
	{
		$this->roomsService = $rooms;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->roomId && !$this->room) {
			$this->setRoom($this->roomId);
		}
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('value', 'default.value')->setRequired()->setMaxLength(255);
		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		if (($this->room && strtolower($this->room->value) != strtolower($values->value) || !$this->room) && $this->roomsService->getByValue($values->value)) {
			$form->addError('realEstates.roomForm.roomExists');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->room) {
				$room         = $this->room;
				$flashMessage = 'realEstates.roomForm.edited';
			} else {
				$room         = new Room($values->value);
				$flashMessage = 'realEstates.roomForm.added';
			}

			$room->value = $values->value;

			$this->em->persist($room)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setRoom($id)
	{
		$this->room   = $this->roomsService->get($id);
		$this->roomId = $id;

		if ($this->room) {
			$this['form']->setDefaults([
				                           'value' => $this->room->value,
			                           ]);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/**
	 * @return Room
	 */
	public function getRoom() { return $this->room; }
}
