<?php declare(strict_types = 1);

namespace RealEstates\AdminModule\Presenters;

use Nette\Http\IResponse;
use RealEstates\AdminModule\Components\Author\IAuthorFormFactory;
use RealEstates\AdminModule\Components\Author\IAuthorsGridFactory;
use RealEstates\AdminModule\Components\Filter\IFilterFormFactory;
use RealEstates\AdminModule\Components\Filter\IFiltersGridFactory;
use RealEstates\AdminModule\Components\Param\IParamFormFactory;
use RealEstates\AdminModule\Components\Param\IParamsGridFactory;
use RealEstates\AdminModule\Components\PropertyType\IPropertyTypeFormFactory;
use RealEstates\AdminModule\Components\PropertyType\IPropertyTypesGridFactory;
use RealEstates\AdminModule\Components\Room\IRoomFormFactory;
use RealEstates\AdminModule\Components\Room\IRoomsGridFactory;
use RealEstates\AdminModule\Components\ISettingsFormFactory;
use RealEstates\AdminModule\Components\Variant\IVariantFormFactory;
use RealEstates\AdminModule\Components\Variant\IVariantsGridFactory;
use RealEstates\AdminModule\Model\Settings;
use RealEstates\Model\Entities\Param;
use RealEstates\Model\Entities\PropertyType;
use RealEstates\Model\Entities\Variant;

class SettingsPresenter extends BasePresenter
{
	/** @var Settings @inject */
	public $settingsService;

	protected function startup()
	{
		parent::startup();
		$this->setHeader('realEstates.title.settings', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.settings'));
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		$this['navigation']->setData(['header' => [
			[
				'title' => 'realEstates.menu.addPropertyType',
				'link'  => 'Settings:addPropertyType',
				'ico'   => 'plus',
			], [
				'title' => 'realEstates.menu.addParam',
				'link'  => 'Settings:addParam',
				'ico'   => 'plus',
			], [
				'title' => 'realEstates.menu.addVariant',
				'link'  => 'Settings:addVariant',
				'ico'   => 'plus',
			], [
				'title' => 'realEstates.menu.addAuthor',
				'link'  => $this->link('authorForm!'),
				'ico'   => 'fas fa-user-plus',
				'class' => 'ajax',
			], [
				'title' => 'realEstates.menu.addRoom',
				'link'  => $this->link('roomForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'realEstates.menu.addFilter',
				'link'  => $this->link('filterForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'realEstates.menu.settings',
				'link'  => $this->link('settingsForm!'),
				'ico'   => 'fas fa-cogs',
				'class' => 'ajax',
			],
		]]);

		$this->template->catalogDownloads = $this->settingsService->get('catalog_downloads');
	}

	public function actionAddPropertyType()
	{
		$this->setHeader('realEstates.title.addPropertyType', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.addPropertyType'));
	}

	public function actionEditPropertyType($id)
	{
		$type = $this->em->getRepository(PropertyType::class)->find($id);

		if (!$type)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader('realEstates.title.editPropertyType', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.editPropertyType'), $type->title);
	}

	public function actionAddParam()
	{
		$this->setHeader('realEstates.title.addParam', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.addParam'));
	}

	public function actionEditParam($id)
	{
		$param = $this->em->getRepository(Param::class)->find($id);

		if (!$param)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader('realEstates.title.editParam', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.editParam'), $param->title);
	}

	public function actionAddVariant()
	{
		$this->setHeader('realEstates.title.addVariant', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.addVariant'));
	}

	public function actionEditVariant($id)
	{
		$variant = $this->em->getRepository(Variant::class)->find($id);

		if (!$variant)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setHeader('realEstates.title.editVariant', 'fas fa-cogs');
		$this->setTitle($this->translator->translate('realEstates.title.editVariant'), $variant->title);
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleSettingsForm()
	{
		$this->template->modal = 'settingsForm';
		$this->redrawControl('modal');
	}

	public function handleAuthorForm($id = null)
	{
		$this->template->modal = 'authorForm';
		$this->redrawControl('modal');

		$control = $this['authorForm'];
		if ($id) {
			$control->setAuthor($id);

			$this->setTitle($this->t('realEstates.title.editAuthor'), $control->getAuthor()->name);
		} else {
			$this->setTitle($this->t('realEstates.title.addAuthor'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleRoomForm($id = null)
	{
		$this->template->modal = 'roomForm';
		$this->redrawControl('modal');

		$control = $this['roomForm'];
		if ($id) {
			$control->setRoom($id);

			$this->setTitle($this->t('realEstates.title.editRoom'), $control->getRoom()->value);
		} else {
			$this->setTitle($this->t('realEstates.title.addRoom'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleFilterForm($id = null)
	{
		$this->template->modal = 'filterForm';
		$this->redrawControl('modal');

		$control = $this['filterForm'];
		if ($id) {
			$control->setFilter($id);

			$this->setTitle($this->t('realEstates.title.editFilter'), $control->getFilter()->value);
		} else {
			$this->setTitle($this->t('realEstates.title.addFilter'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleResetCatalogDownloads()
	{
		if ($this->settingsService->setData('catalog_downloads', 0)) {
			$this->flashMessageSuccess('realEstates.settings.catalogDownloadsReseted');
		} else {
			$this->flashMessageDanger('realEstates.settings.catalogDownloadsResetFailed');
		}

		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentPropertyTypesGrid(IPropertyTypesGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentPropertyTypeForm(IPropertyTypeFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setPropertyType($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redirect('this');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function() {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentParamsGrid(IParamsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentParamForm(IParamFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setParam($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redirect('this');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function() {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentAuthorsGrid(IAuthorsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentAuthorForm(IAuthorFormFactory $factory)
	{
		$control = $factory->create();

		$control['form']->setAjax();

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['authorsGrid']['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['authorsGrid']['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentRoomsGrid(IRoomsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentRoomForm(IRoomFormFactory $factory)
	{
		$control = $factory->create();

		$control['form']->setAjax();

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['roomsGrid']['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['roomsGrid']['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentSettingsForm(ISettingsFormFactory $factory)
	{
		$control = $factory->create();

		$control['form']->setAjax();

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentVariantsGrid(IVariantsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentVariantForm(IVariantFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setVariant($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redirect('this');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function() {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentFiltersGrid(IFiltersGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFilterForm(IFilterFormFactory $factory)
	{
		$control = $factory->create();

		$control['form']->setAjax();

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['filtersGrid']['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['filtersGrid']['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
