<?php declare(strict_types = 1);

namespace RealEstates\FrontModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\BadRequestException;
use Nette\Application\Responses\FileResponse;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\InvalidArgumentException;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use RealEstates\FrontModule\Model\Propertys;
use RealEstates\FrontModule\Model\Dao\Property;
use RealEstates\Model\PropertysPdf;

class PdfRequestForm extends BaseControl
{
	const TYPE_CALCULATION = 'calculation';
	const TYPE_CATALOG     = 'catalog';

	/** @var Propertys */
	protected $propertysService;

	/** @var PropertysPdf */
	protected $propertysPdf;

	/** @var Request */
	protected $httpRequest;

	/** @var Response */
	protected $httpResponse;

	/** @var Property */
	protected $property;

	/** @var int @persistent */
	public $propertyId;

	/** @var string @persistent */
	public $type;

	/** @var string @persistent */
	public $showModal;

	/** @var string */
	public $btnText;

	/** @var IMailer */
	protected $mailer;

	/** @var ITemplateFactory */
	protected $templateFactory;

	/** @var Message */
	protected $message;

	/** @var array */
	protected $params;

	/** @var int */
	protected $id;

	public function __construct($params, IMailer $mailer, ITemplateFactory $templateFactory, Propertys $propertys, PropertysPdf $propertysPdf, Request $request, Response $response)
	{
		$this->propertysService = $propertys;
		$this->propertysPdf     = $propertysPdf;
		$this->httpRequest      = $request;
		$this->httpResponse     = $response;

		$this->params          = $params;
		$this->message         = new Message();
		$this->mailer          = $mailer;
		$this->templateFactory = $templateFactory;

		$this->message->setFrom($params['fromEmail'] ?? '', $params['fromName'] ?? '');
		$this->message->addTo($params['toEmail'] ?? '', $params['toName'] ?? '');

		if (isset($params['bcc'])) {
			foreach (explode(',', $params['bcc']) as $v) {
				$this->message->addBcc(trim($v));
			}
		}
	}

	public function render($args = [])
	{
		if (isset($args['btnText']))
			$this->setBtnText($args['btnText']);
		if (isset($args['propertyId']))
			$this->setType(self::TYPE_CALCULATION, $args['propertyId']);
		else
			$this->setType(self::TYPE_CATALOG);

		$this->template->btnText    = $this->btnText;
		$this->template->property   = $this->property;
		$this->template->formFilled = $this->httpRequest->getCookie('pdfRequestForm');
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== handle
	 */

	public function handleShowForm($id = null)
	{
		if ($id)
			$this->setType(self::TYPE_CALCULATION, $id);
		else
			$this->setType(self::TYPE_CATALOG);

		$this->id                             = $id;
		$this->template->showModal            = true;
		$this->getPresenter()->payload->modal = 'pdfRequestForm';
		$this->redrawControl('modal');
	}

	public function handleDownload($id = null)
	{
		if ($id)
			$this->setType(self::TYPE_CALCULATION, $id);

		$this->id = $id;

		switch ($this->type) {
			case self::TYPE_CALCULATION:
				$file     = $this->propertysPdf->downloadCalculation($this->propertyId);
				$property = $this->propertysService->getPublished($id, null, null, true);
				$name     = $this->translator->translate('front.realEstates.calculationTitle', null, ['name' => $property->title]);
				break;
			case self::TYPE_CATALOG:
			default:
				$file = $this->propertysPdf->downloadCatalog();
				$name = $this->translator->translate('front.realEstates.pdfCatalogTitle');
				break;
		}

		$response = new FileResponse($file, $name);
		$this->getPresenter()->sendResponse($response);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'realEstates.pdfRequestForm.name')->setPlaceholder('realEstates.pdfRequestForm.name')->setRequired();
		$form->addEmail('email', 'realEstates.pdfRequestForm.email')->setPlaceholder('realEstates.pdfRequestForm.email')->setRequired();
		$form->addText('phone', 'realEstates.pdfRequestForm.phone')->setPlaceholder('realEstates.pdfRequestForm.phone');
		$form->addText('location', 'realEstates.pdfRequestForm.location')->setPlaceholder('realEstates.pdfRequestForm.location');

		$form->addSubmit('submit', 'realEstates.pdfRequestForm.submit');
		$form->addHidden('id', $this->id);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$this->httpResponse->setCookie('pdfRequestForm', '1', new DateTime('+2 days'));
			$id = $values->id;

			$template = $this->templateFactory->createTemplate();
			$template->setFile(__DIR__ . '/email.latte');
			$template->setParameters((array) $values);

			switch ($this->type) {
				case self::TYPE_CALCULATION:
					$property = $this->propertysService->getPublished($id, null, null, true);
					$name     = $this->translator->translate('front.realEstates.calculationTitle', null, ['name' => $property->title]);
					break;
				case self::TYPE_CATALOG:
				default:
					$name = $this->translator->translate('front.realEstates.pdfCatalogTitle');
					break;
			}
			$template->setParameters(['subject' => $name]);

			$this->message->setSubject($values->propertyTitle);
			$this->message->setHtmlBody($template->renderToString());

			$this->mailer->send($this->message);

			$this->template->okMessage = $this->t('realEstates.pdfRequestForm.formOk');
			$this->redrawControl('form');
		} catch (\Exception $e) {
			$form->addError($this->t('realEstates.pdfRequestForm.downloadError'));

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setType($type, $propertyId = null)
	{
		switch ($type) {
			case self::TYPE_CALCULATION:
				$property = $this->propertysService->getPublished($propertyId);
				if ($property) {
					$this->property   = $property;
					$this->propertyId = $propertyId;
					$this->type       = $type;
				} else
					throw new BadRequestException();
				break;
			case self::TYPE_CATALOG:
				$this->property = null;
				$this->type     = $type;
				break;
			default:
				throw new InvalidArgumentException();
		}
	}

	public function setBtnText($text)
	{
		$this->btnText = $text;
	}
}
