<?php declare(strict_types = 1);

namespace RealEstates\FrontModule\Components;

use Core\Model\UI\BaseControl;
use RealEstates\FrontModule\Model\Propertys;
use RealEstates\FrontModule\Model\PropertyTypes;
use RealEstates\Model\Entities\Property;

class PropertysFilter extends BaseControl
{
	/** @var Propertys */
	protected $propertysService;

	/** @var PropertyTypes */
	protected $propertyTypesService;

	/** @var Property[] */
	protected $propertys;

	/** @var PropertysList */
	protected $propertysListControl;

	/** @var array */
	protected $allFilters;

	/** @var array */
	protected $typeData;

	/** @var string @persistent */
	public $type;

	/** @var string @persistent */
	public $rooms;

	/** @var string @persistent */
	public $price;

	/** @var string @persistent */
	public $usableArea;

	/** @var string @persistent */
	public $more;

	public function __construct(Propertys $propertys, PropertyTypes $propertyTypes)
	{
		$this->propertysService     = $propertys;
		$this->propertyTypesService = $propertyTypes;
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		foreach ($this->getAllFilters()['price'] as $k => $p) {
			if ($this->price == $p) {
				$this->price = $k;
				break;
			}
		}

		if (count($this->getRoomsFilter()) > 1 || count($this->getMoreFilter()) > 1)
			$presenter['meta']->setMeta('robots', 'noindex, nofollow');

		$presenter->setTitle($this->getHeadTitle());
	}

	public function render()
	{
		$this->template->activeType       = $this->type;
		$this->template->activeRooms      = $this->rooms;
		$this->template->activePrice      = $this->price;
		$this->template->activeUsableArea = $this->usableArea ? explode('-', $this->usableArea) : [];
		$this->template->activeMore       = $this->more;
		$this->template->filters          = $this->getAllFilters();
		$this->template->pageTitle        = $this->getPageTitle();
		$this->template->pageDescription  = $this->getPageDescription();
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Action
	 */

	public function handleType($type = null)
	{
		$this->type = $type;
	}

	public function handleRooms($rooms = null)
	{
		$this->rooms = $rooms;
	}

	public function handlePrice($price = null)
	{
		$this->price = $price;
	}

	public function handleMore($more = null)
	{
		$this->more = $more;
	}

	public function handleUsableArea($usableArea)
	{
		$this->usableArea = $usableArea;
	}

	/*******************************************************************************************************************
	 * ============================== SEO
	 */

	public function getHeadTitle()
	{
		return str_replace('<sup>2</sup>', '²', trim($this->getPageTitle() . ' ' . $this->getAllFiltersString()));
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	protected function getPropertyType()
	{
		if (!$this->typeData)
			$this->typeData = $this->propertyTypesService->getPublished($this->type ?: 1);

		return $this->typeData;
	}

	protected function getPageTitle() { return $this->getPropertyType()['seoTitle']; }

	protected function getPageDescription()
	{

		return str_replace('{$filterString}', strip_tags($this->getAllFiltersString(), '<sup>'), $this->getPropertyType()['seoDescription']);
	}

	private function getAllFiltersString()
	{
		$filter = [];

		if ($this->rooms)
			$filter[] = $this->propertysService->filterValueToText('rooms', $this->getRoomsFilter());
		if ($this->price)
			$filter[] = $this->propertysService->filterValueToText('price', $this->getPriceFilter());
		if ($this->more)
			$filter = array_merge($filter, $this->propertysService->filterValueToText('more', $this->getMoreFilter(), false));
		if ($this->usableArea)
			$filter[] = $this->propertysService->filterValueToText('usableArea', $this->getUsableAreaFilter());

		if ($filter)
			$filter = ' ' . implode(' ', array_map(function($f) { return lcfirst($f); }, $filter));
		else
			$filter = '';

		return $filter;
	}

	public function setPropertysListControl(PropertysList $propertysList)
	{
		$this->propertysListControl = $propertysList;
	}

	public function getAllFilters()
	{
		if (!$this->allFilters) {
			$this->allFilters = $this->propertysService->getAllFilters($this->propertysService->getPublished());
		}

		return $this->allFilters;
	}

	public function createHandleParams($key, $value)
	{
		$r    = [];
		$skip = null;
		$arr  = [$value];

		foreach ($this->{'get' . ucfirst($key) . 'Filter'}() as $v) {
			if ($value != $v)
				$arr[] = $v;
			else
				$skip = $value;
		}

		foreach ($this->getAllFilters()[$key] as $k => $v)
			if (in_array($k, $arr) && $k != $skip)
				$r[$k] = $k;

		return implode('|', $r);
	}

	public function getTypeFilter() { return $this->type; }

	public function getRoomsFilter() { return $this->rooms ? explode('|', (string) $this->rooms) : null; }

	public function getPriceFilter() { return $this->price; }

	public function getMoreFilter() { return $this->more ? explode('|', (string) $this->more) : null; }

	public function getUsableAreaFilter() { return $this->usableArea; }
}
