<?php declare(strict_types = 1);

namespace RealEstates\Model\Entities;

use Core\Model\Entities\TSeo;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

/**
 * Class Property
 * @package RealEstates\Model\Entities
 *
 * @ORM\Table(name="real_estates__property")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\SortableRepository")
 * @ORM\EntityListeners({"RealEstateListener"})
 */
class Property
{
	use Identifier;
	use MagicAccessors;
	use TSeo;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255, nullable=false)
	 */
	protected $alias;

	/**
	 * @var PropertyType
	 * @ORM\ManyToOne(targetEntity="PropertyType")
	 * @ORM\JoinColumn(name="type_id", referencedColumnName="id")
	 */
	public $type;

	/**
	 * @var Room
	 * @ORM\ManyToOne(targetEntity="Room", inversedBy="propertys")
	 * @ORM\JoinColumn(name="room_id", referencedColumnName="id")
	 */
	public $rooms;

	/**
	 * @var float
	 * @ORM\Column(name="ground_floor_area", type="float", nullable=true, options={"unsigned":true})
	 */
	protected $groundFloorArea;

	/**
	 * @var float
	 * @ORM\Column(name="garret_area", type="float", nullable=true, options={"unsigned":true})
	 */
	protected $garretArea;

	/**
	 * @var float
	 * @ORM\Column(name="garage_area", type="float", nullable=true, options={"unsigned":true})
	 */
	protected $garageArea;

	/**
	 * @var int
	 * @ORM\Column(name="modify_available", type="smallint", options={"default":1})
	 */
	public $modifyAvailable;

	/**
	 * @var Author
	 * @ORM\ManyToOne(targetEntity="Author", inversedBy="propertys")
	 * @ORM\JoinColumn(name="author_id", referencedColumnName="id")
	 */
	public $author;

	/**
	 * @var Filter[]
	 *
	 * @ORM\ManyToMany(targetEntity="Filter", inversedBy="propertys", indexBy="id")
	 * @ORM\JoinTable(name="real_estates__property_filters",
	 *  joinColumns={@ORM\JoinColumn(name="property_id", referencedColumnName="id")},
	 *  inverseJoinColumns={@ORM\JoinColumn(name="filter_id", referencedColumnName="id")})
	 */
	protected $filters;

	/**
	 * @var array
	 * @ORM\Column(name="simple_params", type="array", nullable=true)
	 */
	public $simpleParams;

	/**
	 * @var string
	 * @ORM\Column(name="introtext", type="string", length=255, nullable=true)
	 */
	protected $introtext;

	/**
	 * @var string
	 * @ORM\Column(name="text", type="text", nullable=true)
	 */
	protected $text;

	/**
	 * @var Album
	 * @ORM\OneToOne(targetEntity="Gallery\Model\Entities\Album")
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $gallery;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_up", type="datetime", nullable=true)
	 */
	protected $publishUp;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_down", type="datetime", nullable=true)
	 */
	protected $publishDown;

	/**
	 * @var int
	 *
	 * @Gedmo\SortableGroup
	 * @ORM\Column(name="is_offline", type="smallint", options={"default":0})
	 */
	public $isOffline;

	/**
	 * @var string
	 * @ORM\Column(name="views", type="integer", options={"default":0,"unsigned":true})
	 */
	public $views;

	/**
	 * @var PropertyParam[]
	 * @ORM\OneToMany(targetEntity="PropertyParam", mappedBy="property")
	 */
	private $params;

	/**
	 * @var array
	 * @ORM\Column(name="attribs", type="array", nullable=true)
	 */
	public $attribs;

	/**
	 * @var int
	 *
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	private $position;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var int
	 * @ORM\Column(name="pdf_downloads", type="integer", options={"default":0})
	 */
	public $pdfDownloads;

	/**
	 * Property constructor.
	 *
	 * @param $title
	 */
	public function __construct($title)
	{
		$this->title = $title;
		$this->setAlias($title);
		$this->modifyAvailable = 1;
		$this->isPublished     = 1;
		$this->isOffline       = 0;
		$this->views           = 0;
		$this->params          = new ArrayCollection();
		$this->filters         = new ArrayCollection();
		$this->pdfDownloads    = 0;
		$this->seo             = [];
		$this->setPosition(-1);
	}

	public function getModified() { return $this->modified; }

	/*****
	 * === Alias
	 */

	public function setAlias($alias = null)
	{
		if ($alias)
			$this->alias = Strings::webalize($alias);
		else
			$this->alias = Strings::webalize($this->title);
	}

	/*****
	 * === Params
	 */

	public function getParams() { return $this->params; }

	public function addParam($param)
	{
		if (!$this->params->contains($param)) {
			$this->params->add($param);
		}
	}

	/*****
	 * === Area
	 */

	public function getUsableArea() { return $this->groundFloorArea + $this->garretArea; }

	public function getBuildArea() { return $this->groundFloorArea + $this->garageArea; }

	public function setGroundFloorArea($area) { $this->groundFloorArea = is_float($area) || is_numeric($area) ? $area : null; }

	public function setGarretArea($area) { $this->garretArea = is_float($area) || is_numeric($area) ? $area : null; }

	public function setGarageArea($area) { $this->garageArea = is_float($area) || is_numeric($area) ? $area : null; }

	/*****
	 * === Attribs
	 */

	public function getAttribs() { return $this->attribs ?: []; }

	public function getAttr($key) { return $this->attribs[$key] ?? null; }

	public function setAttr($key, $value)
	{
		if ($value == null) {
			unset($this->attribs[$key]);
		} else {
			$this->attribs[$key] = $value;
		}
	}

	/*****
	 * === Filters
	 */

	public function getFilters() { return $this->filters; }

	public function setFilters(array $filters)
	{
		foreach ($this->filters as $id => $filter) {
			if (!isset($filters[$id])) {
				$this->filters->remove($id);
			} else {
				unset($filters[$id]);
			}
		}

		foreach ($filters as $id => $filter) {
			$this->filters[$id] = $filter;
		}
	}

	/******
	 * === Position
	 */

	public function setPosition($position)
	{
		$this->position = $position;
	}

	public function getPosition()
	{
		return $this->position;
	}
}
