<?php declare(strict_types = 1);

namespace RealEstates\Model\Navigation;

use Core\Model\Dao\SiteMapImage;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Providers\INavigationItem;
use Nette\Http\Url;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use RealEstates\FrontModule\Model\Propertys;
use RealEstates\FrontModule\Model\Rooms;
use RealEstates\Model\ConfigOptions;

class Home extends BaseNavigation implements INavigationItem
{
	protected $title           = 'home';
	protected $presenter       = 'RealEstates:Front:Default';
	protected $action          = 'default';
	protected $presenterDetail = 'RealEstates:Front:Default';
	protected $actionDetail    = 'detail';

	/** @var ConfigOptions */
	protected $configOptions;

	/** @var array */
	protected $filterKeys = ['type', 'rooms', 'roomsmp', 'price', 'more', 'moremp', 'usableArea'];

	/** @var string */
	protected $filterComponentName = 'propertysFilter';

	/** @var Propertys */
	protected $propertysService;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Rooms */
	protected $roomsService;

	public function __construct(ConfigOptions $configOptions, Propertys $propertys, TemplatePages $templateBlocks, Rooms $rooms)
	{
		$this->configOptions        = $configOptions;
		$this->propertysService     = $propertys;
		$this->templatePagesService = $templateBlocks;
		$this->roomsService         = $rooms;
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer()
	{
		$container = new BaseContainer();

		$templates = [];
		foreach ($this->templatePagesService->getEr()->findBy(['isPublished' => '1'], ['title' => 'ASC']) as $t)
			$templates[$t->getId()] = $t->title;
		$container->addSelect('templatePage', 'realEstates.navigation.home.template', $templates);
		$container->addText('defaultTypeUrl', 'realEstates.navigation.home.defaultTypeUrl');

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter == $presenter && $this->action == $action)
			return 1;
		if ($this->presenterDetail == $presenter && $this->actionDetail == $action)
			return 2;

		return false;
	}

	//TODO otestovat otevření offline domu
	public function routerIn($navigation, $urlParams = [])
	{
		$returnDefault = [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];

		if (isset($urlParams['do']))
			$returnDefault['do'] = $urlParams['do'];

		if (isset($urlParams['path']) && $navigation->alias != $urlParams['path']) {
			$params    = explode('/', $urlParams['path']);
			$lastParam = end($params);
			$id        = explode('-', $lastParam, 2);

			// TODO udělat něco aby se nemusel vždy tahat komplet domy
			$propertys = $this->propertysService->getPublished();
			$filters   = $this->propertysService->getAllFilters($propertys);

			// Nastavení filtru z url
			foreach ($this->filterKeys as $key) {
				if (substr($key, -2) == 'mp' && $urlParams[$this->filterComponentName . '-' . $key]) {
					$keyS  = substr($key, 0, -2);
					$value = $urlParams[$this->filterComponentName . '-' . $key];

					$returnDefault[$this->filterComponentName . '-' . $key]  = $value;
					$returnDefault[$this->filterComponentName . '-' . $keyS] = $value;
					continue;
				}

				if (!isset($filters[$key]))
					continue;

				foreach ($filters[$key] as $value) {
					$strpos = strpos($lastParam, Strings::webalize(strip_tags($value)));
					if ($key == 'type' && $strpos === 0 || $key != 'type' && $strpos !== false) {
						$id                                                     = null;
						$returnDefault[$this->filterComponentName . '-' . $key] = $this->propertysService->filterTextToValue($key, $value);
					}
				}
			}

			if ($id != null && $this->configOptions->get('removePropertyIdInUrl')) {
				$property = $this->propertysService->getPublishedByAlias($lastParam);

				if ($property)
					return [
						'presenter' => $this->presenterDetail,
						'action'    => $this->actionDetail,
						'id'        => $property->id,
					];
			} else if (is_numeric($id[0])) {
				return [
					'presenter' => $this->presenterDetail,
					'action'    => $this->actionDetail,
					'id'        => $id[0],
				];
			}
		}

		return $returnDefault;
	}

	public function routerOut($navigation, &$urlParams)
	{
		if ($navigation instanceof Navigation == false)
			return false;

		if (isset($urlParams['id'])) {
			$property = $this->propertysService->getForNavigation($urlParams['id']);
			if ($property) {
				$propertyUrl = $property['alias'];

				if (!$this->configOptions->get('removePropertyIdInUrl'))
					$propertyUrl = $property['id'] . '-' . $propertyUrl;

				return ($navigation->isHomepage ? '' : '/' . $navigation->alias) . '/' . $propertyUrl;
			}
		}

		$pfUrl          = '';
		$pfUrlParams    = [];
		$do             = $urlParams['do'] ?: '';
		$defaultTypeUrl = $navigation->componentParams['defaultTypeUrl'] ?: '';

		if (!isset($urlParams[$this->filterComponentName . '-type']) || $urlParams[$this->filterComponentName . '-type'] === null)
			$pfUrl .= '-' . Strings::webalize($defaultTypeUrl);

		// Vytvoření url z filtru
		foreach ($this->filterKeys as $key) {
			if (substr($key, -2) == 'mp')
				continue;

			$urlKey = $this->filterComponentName . '-' . $key;
			if (isset($urlParams[$urlKey]) && $urlParams[$urlKey] !== null) {

				if (strpos((string) $urlParams[$urlKey], '|') !== false && in_array($key . 'mp', $this->filterKeys)) {
					$pfUrlParams[$urlKey . 'mp'] = $urlParams[$urlKey];
				}

				$pfUrl .= '-' . Strings::webalize(strip_tags($this->propertysService->filterValueToText($key, $urlParams[$urlKey])));
				unset($urlParams[$urlKey]);
			}

			if ($do === $urlKey) {
				unset($urlParams['do']);
			}

			if ($urlParams[$urlKey] === null)
				unset($urlParams[$urlKey]);

			if (isset($urlParams[$urlKey]) && $urlParams[$urlKey] === null) {
				unset($urlParams[$urlKey]);
			}
		}

		$urlParams = array_merge($urlParams, $pfUrlParams);

		if ($pfUrl) {
			$pfUrl = '/' . ltrim($pfUrl, '-');
		}

		return $navigation->isHomepage ? $pfUrl : '/' . $navigation->alias . $pfUrl;
	}

	public function getComponentChild($navigation = null, $siteMapData = false)
	{
		$arr = [];

		foreach ($this->propertysService->getPublished() as $property) {
			$data = [
				'action'    => $this->actionDetail,
				'id'        => $property->id,
				'presenter' => $this->presenterDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($property->getModified())
					$siteMapUrl->setModified($property->getModified());
				$siteMapUrl->setSeo($property->getSeo());

				if ($property->getAlbum())
					foreach (array_merge([$property->getAlbum()->cover], $property->getAlbum()->getImages()) as $img) {
						$siteMapImage           = new SiteMapImage();
						$siteMapImage->filePath = $img->file;
						$siteMapImage->caption  = $img->description;
						$siteMapImage->title    = $img->title;

						$siteMapUrl->addImage($siteMapImage);
					}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			if ($navigation)
				$data['activeNavigation'] = $navigation;

			$arr[] = $data;
		}

		return $arr;
	}
}
