<?php declare(strict_types = 1);

namespace References\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use References\Model\Entities\General;
use References\Model\GeneralGroups;
use References\Model\GeneralReferences;

class GeneralForm extends TemplateReaderControl
{
	/** @var General */
	public $entity;

	/** @var GeneralReferences */
	protected $referencesServices;

	/** @var GeneralGroups */
	protected $groupsService;

	public function __construct(GeneralReferences $references, GeneralGroups $groups)
	{
		$this->referencesServices = $references;
		$this->groupsService      = $groups;
	}

	public function render()
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->referencesServices->getTemplatesDir());
		$this->templateReader->setTranslateKey('generalReferences');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$groups = [];
		foreach ($this->groupsService->getEr()->findBy([], ['title' => 'ASC']) as $g) {
			$arr = [];
			$t   = $g;
			while ($t->parent) {
				$t     = $t->parent;
				$arr[] = $t->title;
			}
			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}
		asort($groups);

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addFilesManager('image', 'default.image');
		$form->addEditor('introtext', 'default.introtext')->setMaxLength(255)
			->setToolbar('Text')->setHeight(100)->setDisableAutoP();
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addBool('isFeatured', 'default.isFeatured')->setDefaultValue(0);
		$form->addLangsSelect('lang', 'default.language');
		$form->addSelect('group', 'default.group', $groups);
		$form->addSelect('template', $this->t('default.templates'), $templates)->setPrompt($this->t('default.choose'))
			->setTranslator(null)->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->entity) {
				$reference    = $this->entity;
				$flashMessage = 'references.referenceForm.edited';
			} else {
				$reference    = new General($values->title, $values->template);
				$flashMessage = 'references.referenceForm.added';
			}

			$reference->title       = $values->title;
			$reference->image       = $values->image;
			$reference->lang        = $values->lang;
			$reference->introtext   = $values->introtext;
			$reference->isPublished = $values->isPublished;
			$reference->isFeatured  = $values->isFeatured;
			$reference->template    = $values->template;
			$reference->setTexts($form->getComponent('component')->getValues(true));

			$reference->group = $this->groupsService->getReference($values->group);

			$this->em->persist($reference)->flush();
			$form->addCustomData('referenceId', $reference->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setReference($id)
	{
		$this->entity = $this->referencesServices->get($id);

		if ($this->entity) {
			$form = $this['form'];
			$r    = $this->entity;
			$form->setDefaults([
				'title'       => $r->title,
				'image'       => $r->image,
				'lang'        => $r->lang,
				'isFeatured'  => $r->isFeatured,
				'introtext'   => $r->introtext,
				'isPublished' => $r->isPublished,
			]);


			if ($r->template && array_key_exists($r->template, $form['template']->getItems())) {
				$form['template']->setDefaultValue($r->template);

				if ($this->httpRequest->getQuery('do') != 'templatePageForm-loadInputs') {
					$this->templateReader->loadTemplateComponents($form['component'], $this->httpRequest->getPost('template') ?: $r->template);
					$this->templateReader->setDefaults($form['component'], $r);
				}
			}

			if ($r->group && array_key_exists($r->group->getId(), $form['group']->getItems())) {
				$form['group']->setDefaultValue($r->group->getId());
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
