<?php declare(strict_types = 1);

namespace References\AdminModule\Components;

use Core\Model\UI\BaseControl;
use References\Model\Entities\General;
use References\Model\GeneralGroups;
use References\Model\GeneralReferences;

class GeneralsGrid extends BaseControl
{
	/** @var GeneralReferences */
	protected $referencesService;

	/** @var GeneralGroups */
	protected $groupsService;

	public function __construct(GeneralReferences $references, GeneralGroups $groups)
	{
		$this->referencesService = $references;
		$this->groupsService     = $groups;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->referencesService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->referencesService->getEr()->createQueryBuilder('r')->addSelect('g')
			->join('r.group', 'g')->addOrderBy('g.title', 'ASC')->addOrderBy('r.position', 'ASC');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('generalsGrid:gridSortableRow!');
		$grid->setSortableGroup('group');

		// Column
		$grid->addColumnLink('title', 'default.title', 'General:editReference');
		$grid->addColumnText('group', 'default.group', 'group.title');
		$grid->addColumnText('lang', 'default.lang');
		$grid->addColumnStatus('isFeatured', 'default.top')->setAlign('center')
			->addOption(1, 'default.setFeatured')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.setUnfeatured')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridFeaturedChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$groups = ['' => ''];
		foreach ($this->groupsService->getAll() as $group)
			$groups[$group->getId()] = $group->title;
		$grid->addFilterSelect('group', '', $groups, 'g.id');

		// Actions
		$grid->addAction('edit', '', 'General:editReference')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		return $grid;
	}

	public function gridFeaturedChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->referencesService->setFeatured($id, $newStatus))
			$presenter->flashMessageSuccess('default.featuredChanged');
		else
			$presenter->flashMessageDanger('default.featuredChangedFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->referencesService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);
		bdump($position);

		if ($id != null && $position != null && $this->referencesService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
	}

}
