<?php declare(strict_types = 1);

namespace References\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use References\Model\Entities\General;
use References\FrontModule\Model\Dao as DaoReferences;

/**
 * Class GeneralReferences
 * @package References\FrontModule\Model
 *
 * @method General|object|null getReference($id)
 * @method General[]|null getAll()
 */
class GeneralReferences extends BaseEntityService
{
	protected $entityClass = General::class;

	public function getAll()
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r')->addSelect('g')->where('r.isPublished = 1')
			->innerJoin('r.group', 'g')->andWhere('g.isPublished = 1')->getQuery()->getResult();

		return $this->fillDao($referencesRaw);
	}

	public function get($id)
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r')->addSelect('g')->where('r.isPublished = 1')
			->andWhere(is_array($id) ? 'r.id IN (:ids)' : 'r.id = :ids')->setParameter('ids', $id)
			->innerJoin('r.group', 'g')->andWhere('g.isPublished = 1')->getQuery()->getResult();

		$references = $this->fillDao($referencesRaw);

		return is_array($id) ? $references : ($references[$id] ?? null);
	}

	/**
	 * @param $groupId
	 *
	 * @return General[]|null
	 */
	public function getPublishedByGroup($groupId, $limit = null, $featuredOnly = false)
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r', 'r.id')->addSelect('g')->where('r.isPublished = 1')
			->innerJoin('r.group', 'g')->andWhere(is_array($groupId) ? 'g.id IN (:groupId)' : 'g.id = :groupId')->andWhere('g.isPublished = 1')
			->setParameter('groupId', $groupId)->orderBy('r.position', 'ASC');

		if ($featuredOnly)
			$referencesRaw->andWhere('r.isFeatured = 1');

		$referencesRaw = $referencesRaw->getQuery();

		if ($limit)
			$referencesRaw->setMaxResults($limit);

		$referencesRaw = $referencesRaw->useResultCache(true, 60)->getResult();

		/** @var General[] $referencesRaw */
		return $this->fillDao($referencesRaw);
	}

	/**
	 * @param General[] $referencesRaw
	 *
	 * @return array
	 */
	private function fillDao($referencesRaw)
	{
		$references = [];
		$groups     = [];

		foreach ($referencesRaw as $r) {
			$reference = (new DaoReferences\GeneralReference())
				->setId($r->getId())
				->setTitle($r->title)
				->setImage($r->image)
				->setIsFeatured($r->isFeatured)
				->setIntrotext($r->introtext)
				->setParams($r->getTexts());

			if (!isset($groups[$r->group->getId()])) {
				$group = (new DaoReferences\Group())
					->setId($r->group->getId())
					->setTitle($r->group->title);

				$groups[$group->id] = $group;
			}

			$reference->setGroup($groups[$r->group->getId()]);

			$references[$reference->id] = $reference;
		}

		return $references;
	}
}