<?php declare(strict_types = 1);

namespace References\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Nette\Utils\Html;
use References\Model\GeneralGroups;

class GeneralGroupsGrid extends BaseControl
{
	/** @var GeneralGroups */
	protected $groupsService;

	public function __construct(GeneralGroups $groups)
	{
		$this->groupsService = $groups;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->groupsService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setSortable();
		$grid->setSortableHandler('generalGroupsGrid:gridSortableRow!');

		$qb = $this->groupsService->getEr()->createQueryBuilder('g')
			->orderBy('g.root')->addOrderBy('g.lft');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('title', 'default.title', 'General:editGroup')->setRenderer(function($row) {
			$title = ' ' . $row->title;
			for ($i = 0; $i < $row->getLevel(); $i++)
				$title = '---' . $title;

			return Html::el('a', ['href' => $this->getPresenter()->link('General:editGroup', $row->getId())])->setText(trim($title));
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter

		// Actions
		$grid->addAction('edit', '', 'General:editGroup')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		$grid->setRowCallback(function($row, $tr) {
			$groupId = ($row->parent ? $row->parent->getId() : '0') . '-' . $row->getLevel();
			/** @var Html $tr */
			$tr->addClass('group_' . $groupId);
			$tr->addAttributes(['data-group-tree' => $groupId]);

			return $tr;
		});

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->groupsService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();

		$id   = $request->getPost('id');
		$move = $request->getPost('move');
		$er   = $this->groupsService->getEr();
		$cat  = $this->groupsService->get($id);

		if ($id && $move && $cat && $cat->getLevel() > 0 && $move !== 0) {
			if ($move < 0)
				$er->moveUp($cat, abs($move));
			else if ($move > 0)
				$er->moveDown($cat, $move);
			$presenter->flashMessageSuccess('default.positionChanged');
		} else
			$presenter->flashMessageDanger('default.positionChangeFailed');

		$this->em->flush();

		$presenter->redrawControl('flashes');
	}
}
