<?php declare(strict_types = 1);

namespace References\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Nette\Localization\ITranslator;
use References\Model\Entities\General;
use References\FrontModule\Model\Dao as DaoReferences;

/**
 * TODO cache
 *
 * Class GeneralReferences
 * @package References\FrontModule\Model
 *
 * @method General|object|null getReference($id)
 */
class GeneralReferences extends BaseEntityService
{
	protected $entityClass = General::class;

	/** @var ITranslator */
	protected $translator;

	/**
	 * GeneralReferences constructor.
	 *
	 * @param ITranslator $translator
	 */
	public function __construct(ITranslator $translator)
	{
		$this->translator = $translator;
	}

	/**
	 * @return array|Dao\GeneralReference[]
	 */
	public function getAll()
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r')->addSelect('g')->where('r.isPublished = 1')
			->addSelect('gp')
			->innerJoin('r.group', 'g')->andWhere('g.isPublished = 1')
			->leftJoin('g.parent', 'gp')
			->orderBy('r.group')->addOrderBy('r.position')->getQuery()->getResult();

		return $this->fillDao($referencesRaw);
	}

	public function get($id)
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r')->addSelect('g')->where('r.isPublished = 1')
			->addSelect('gp')
			->andWhere(is_array($id) ? 'r.id IN (:ids)' : 'r.id = :ids')->setParameter('ids', $id)
			->innerJoin('r.group', 'g')->andWhere('g.isPublished = 1')
			->leftJoin('g.parent', 'gp')
			->getQuery()->getResult();

		$references = $this->fillDao($referencesRaw);

		return is_array($id) ? $references : ($references[$id] ?? null);
	}

	public function getPublishedIdByAlias($alias)
	{
		$id = $this->getEr()->createQueryBuilder('r')->select('r.id')
			->where('r.isPublished = 1')->andWhere('r.alias = :alias')
			->setParameter('alias', $alias)->getQuery()->getOneOrNullResult();

		return $id ? $id['id'] : null;
	}

	/**
	 * @param        $groupId
	 * @param null   $limit
	 * @param bool   $featuredOnly
	 * @param string $groupBy
	 *
	 * @return Dao\GeneralReference|DaoReferences\Group[]
	 */
	public function getPublishedByGroup($groupId, $limit = null, $featuredOnly = false, string $groupBy = null)
	{
		$referencesRaw = $this->getEr()->createQueryBuilder('r', 'r.id')->addSelect('g')->where('r.isPublished = 1')
			->innerJoin('r.group', 'g')->andWhere(is_array($groupId) ? 'g.id IN (:groupId)' : 'g.id = :groupId')->andWhere('g.isPublished = 1')
			->setParameter('groupId', $groupId)
			->addOrderBy('r.position', 'ASC');

		if ($featuredOnly)
			$referencesRaw->andWhere('r.isFeatured = 1');

		$referencesRaw = $referencesRaw->getQuery();

		if ($limit)
			$referencesRaw->setMaxResults($limit);

		$referencesRaw = $referencesRaw->getResult();

		$return = [];
		$tmp    = $this->fillDao($referencesRaw, $groupBy);

		if ($groupBy == 'group') {
			foreach ($groupId as $id) {
				$return[$id] = $tmp[$id];
			}
		} else {
			$return = $tmp;
		}

		return $return;
	}

	/**
	 * @param General[] $referencesRaw
	 * @param string    $groupBy
	 *
	 * @return DaoReferences\GeneralReference[]
	 */
	private function fillDao($referencesRaw, string $groupBy = null)
	{
		$references = [];
		$groups     = [];

		foreach ($referencesRaw as $r) {
			$reference = (new DaoReferences\GeneralReference())
				->setId($r->getId())
				->setTitle($r->title)
				->setAlias($r->alias)
				->setImage($r->image)
				->setIsFeatured($r->isFeatured)
				->setIntrotext($r->introtext)
				->setParams($r->getTexts())
				->setModified($r->getModified());
			$reference->setSeo($r->getSeo());

			if (!isset($groups[$r->group->getId()])) {
				$group = (new DaoReferences\Group())
					->setId($r->group->getId())
					->setTitle($r->group->title);

				if ($r->group->parent) {
					$groupP           = (new DaoReferences\Group())
						->setId($r->group->parent->getId())
						->setTitle($r->group->parent->title);
					$group->parent    = $groupP;
					$groups->childs[] = $group;
				}

				$groups[$group->id] = $group;
			}

			$reference->setGroup($groups[$r->group->getId()]);
			$groups[$r->group->getId()]->items[] = $reference;

			$references[$reference->id] = $reference;
		}

		return $groupBy == 'group' ? $groups : $references;
	}
}
