<?php declare(strict_types = 1);

namespace References\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TLang;
use Core\Model\Entities\TSeo;
use Core\Model\TemplateReader\Entity\ITemplate;
use Core\Model\TemplateReader\Entity\TTemplate;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

/**
 * Class General
 * @package References\Model\Entities
 *
 * @ORM\Table(name="references__general")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\SortableRepository")
 * @ORM\EntityListeners({"GeneralListener"})
 */
class General implements ITemplate
{
	use TId;
	use TTemplate;
	use TLang;
	use TSeo;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255, nullable=false)
	 */
	public $alias;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public $image;

	/**
	 * @var string
	 * @ORM\Column(name="introtext", type="string", nullable=true, length=255)
	 */
	public $introtext;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", nullable=false)
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="is_featured", type="smallint", nullable=false)
	 */
	public $isFeatured;

	/**
	 * @var int
	 *
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	public $position;

	/**
	 * @var GeneralGroup
	 *
	 * @ORM\ManyToOne(targetEntity="GeneralGroup", inversedBy="references")
	 * @ORM\JoinColumn(name="group_id", referencedColumnName="id", nullable=true)
	 */
	public $group;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * TemplatePage constructor.
	 *
	 * @param $title
	 * @param $template
	 */
	public function __construct($title, $template)
	{
		$this->setTitle($title);
		$this->template    = $template;
		$this->isPublished = 1;
		$this->isFeatured  = 0;
		$this->texts       = [];
		$this->seo         = [];
		$this->setPosition(-1);
	}

	/*********
	 * Title
	 */

	/**
	 * @param string $title
	 *
	 * @return $this
	 */
	public function setTitle($title)
	{
		$this->title = $title;
		if (!$this->alias)
			$this->setAlias($title);

		return $this;
	}

	/**
	 * @param string $alias
	 *
	 * @return $this
	 */
	public function setAlias($alias)
	{
		$this->alias = Strings::webalize($alias ? $alias : $this->title);

		return $this;
	}

	public function getModified() { return $this->modified; }

	public function setPosition($position)
	{
		$this->position = intval($position);
	}
}
