<?php declare(strict_types = 1);

namespace SimpleProducts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use SimpleProducts\Model\Categories;
use SimpleProducts\Model\Entities\Product;
use SimpleProducts\Model\SimpleProducts;

class ProductForm extends BaseControl
{
	/** @var Product */
	public $product;

	/** @var SimpleProducts */
	protected $productsService;

	/** @var Categories */
	protected $categoriesService;

	public function __construct(SimpleProducts $products, Categories $categories)
	{
		$this->productsService = $products;
		$this->categoriesService = $categories;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$categories = [];
		foreach ($this->categoriesService->getEr()->findBy([], ['title' => 'ASC']) as $g) {
			$arr = [];
			$t = $g;
			while ($t->parent) {
				$t = $t->parent;
				$arr[] = $t->title;
			}
			$categories[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}
		asort($categories);

		$products = [null => $this->t('simpleProducts.productForm.noAlternative')];

		foreach ($this->productsService->getAll() as $product) {
			$products[$product->getId()] = $product->title;
		}

		$form->addEditor('title', 'default.name2')->setRequired()->setMaxLength(255)->setWidth(500)->setHeight(100);;
		$form->addEditor('description', 'default.description')->setWidth(500)->setHeight(300);
		$form->addFilesManager('image', 'default.image');
		$form
			->addText('price', 'simpleProducts.productForm.productPrice')
			->setDefaultValue(0);
		$form->addText('externalUrl', 'default.url')->addRule(Form::URL);
		$form->addSelect('alternativeOf', 'simpleProducts.productForm.alternativeTo', $products);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('category', 'default.category', $categories);
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->product) {
				$product = $this->product;
				$flashMessage = 'simpleProducts.productForm.edited';
			} else {
				$product = new Product($values->title);
				$flashMessage = 'simpleProducts.productForm.added';
			}

			$product->title = $values->title;
			$product->isPublished = $values->isPublished;
			$product->price = $values->price;
			$product->image = $values->image;
			$product->alternativeOf = $values->alternativeOf ? $this->productsService->get($values->alternativeOf) : null;
			$product->description = $values->description;
			$product->externalUrl = $values->externalUrl;
			$product->categories = $values->category ? array_map(function ($id) {
				return $this->categoriesService->get((int)$id);
			}, $values->category) : [];

			$this->em->persist($product)->flush();
			$this->getPresenter()->flashMessage($flashMessage);


		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			return false;
		}
	}

	public function setProduct($id)
	{
		$this->product = $this->productsService->get($id);

		if ($this->product) {
			$p = $this->product;
			$this['form']->setDefaults([
				'title'         => $p->title,
				'isPublished'   => $p->isPublished,
				'price'         => $p->price,
				'image'         => $p->image,
				'description'   => $p->description,
				'externalUrl'   => $p->externalUrl
			]);

			if(isset($p->alternativeOf))
				$this['form']['alternativeOf']->setDefaultValue($this->productsService->get($p->alternativeOf)->getId());

			foreach ($p->categories as $c) {
				$d = [];
				if (array_key_exists($c->getId(), $this['form']['group']->getItems())) {
					$d[] = $c->getId();
				}

				$this['form']['category']->setDefaultValue($d);
			}
		} else {
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
		}
	}

}
