<?php declare(strict_types=1);

namespace SimpleProducts\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="simple_products__category")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 */
class Category
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255)
	 */
	public $title;

	/**
	 * @var Product
	 * @ORM\ManyToMany(targetEntity="Product", mappedBy="categories")
	 */
	protected $products;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255)
	 */
	protected $alias;

	/** @var string
	 * @ORM\Column(name="layout", type="string", length=60)
	 */
	protected $layout;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var Category[];
	 * @ORM\OneToMany(targetEntity="Category", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	protected $children;

	/**
	 * @var Category
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Category", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	private $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	private $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	private $gt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private $root;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	public function __construct($title)
	{
		$this->title = $title;
		$this->setAlias($title);
		$this->layout = 'default';
		$this->isPublished = 1;
		$this->products = new ArrayCollection();
		$this->children = new ArrayCollection();
	}

	public function setAlias($alias)
	{
		$this->alias = Strings::webalize($alias);
	}

	/**
	 * @return Product
	 */
	public function getProducts()
	{
		return $this->products;
	}

	public function addProduct($product)
	{
		if ($this->products->contains($product)) {
			return true;
		}

		return $this->products->add($product);
	}

	public function removeProduct($product)
	{
		if ($this->products->contains($product)) {
			return $this->products->removeElement($product);
		}
		return true;
	}

	/**
	 * @return mixed
	 */
	public function getRoot()
	{
		return $this->root;
	}

	/**
	 * @param Category $parent
	 */
	public function setParent($parent)
	{
		$this->parent = $parent;
	}

	/**
	 * @return Category
	 */
	public function getParent()
	{
		return $this->parent;
	}

	/**
	 * @return Category[]
	 */
	public function getChildren()
	{
		return $this->children;
	}

	public function getLevel()
	{
		return $this->lvl;
	}
}