<?php declare(strict_types = 1);

namespace SimpleProducts\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;

/**
 * @ORM\Table(name="simple_products__product")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\SortableRepository")
 */
class Product
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public $title;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="float", nullable=true)
	 */
	public $price;

	/**
	 * @var string
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public $description;

	/**
	 * @var string
	 * @ORM\Column(name="url", type="string", length=255, nullable=true)
	 */
	public $externalUrl;

	/**
	 * @var string
	 * @ORM\Column(name="url2", type="string", length=255, nullable=true)
	 */
	protected $externalUrl2;

	/**
	 * @var string
	 * @ORM\Column(name="product_id", type="string", length=255, nullable=true)
	 */
	public $productId;

	/**
	 * @var string
	 * @ORM\Column(name="availability", type="string", length=255, nullable=true)
	 */
	public $availability;

	/**
	 * @var string
	 * @ORM\Column(name="image",type="string",length=255,nullable=true)
	 */
	public $image;

	/**
	 * @var int
	 * @ORM\Column(name="in_stock", type="string", length=255, nullable=true, options={"default":1})
	 */
	public $inStock;
	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var array
	 * @ORM\Column(name="parameters", type="array", nullable=true)
	 */
	public $params;

	/**
	 * @var Product
	 * @ORM\ManyToOne(targetEntity="Product", inversedBy="alternatives")
	 * @ORM\JoinColumn(name="alternative_of_id", referencedColumnName="id")
	 */
	public $alternativeOf;

	/**
	 * @var Product[]
	 * @ORM\OneToMany(targetEntity="Product", mappedBy="alternativeOf")
	 */
	public $alternatives;

	/**
	 * @var Category[]
	 *
	 * @ORM\ManyToMany(targetEntity="Category", inversedBy="categories")
	 * @ORM\JoinTable(name="simple_products__category_product",
	 *  joinColumns={@ORM\JoinColumn(name="product_id", referencedColumnName="id")},
	 *  inverseJoinColumns={@ORM\JoinColumn(name="category_id", referencedColumnName="id")})
	 */
	public $categories;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer", options={"default": 0})
	 */
	protected $position;

	public function __construct($title)
	{
		$this->title        = $title;
		$this->isPublished  = 1;
		$this->categories   = new ArrayCollection();
		$this->params       = [];
		$this->alternatives = new ArrayCollection();
		$this->setPosition(-1);
	}

	public function getParams()
	{
		return $this->params ?: [];
	}

	public function getParam($key)
	{
		return $this->params[$key] ?? null;
	}

	public function setParam($key, $value)
	{
		if ($value == null) {
			unset($this->params[$key]);
		} else {
			$this->params[$key] = $value;
		}
	}

	public function getExternalUrl2()
	{
		return 'https://drjacobs-shop.de/gutschein/check21?articles=' . $this->productId . '&c=checkout';
	}

	public function setPosition($position) { $this->position = intval($position); }

	public function getPosition() { return $this->position; }
}
