<?php declare(strict_types = 1);

namespace Slides\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Slides\Model\Entities\Slide;
use Slides\Model\Entities\SlideGroup;
use Slides\Model\SlideGroups;
use Slides\Model\Slides;

class SlidesGrid extends BaseControl
{
	/** @var Slides */
	protected $slidesService;

	/** @var SlideGroups */
	protected $slideGroupsService;

	public function __construct(Slides $slides, SlideGroups $slideGroups)
	{
		$this->slidesService      = $slides;
		$this->slideGroupsService = $slideGroups;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$groups = $this->slideGroupsService->getAll();

		$qb = $this->slidesService->getEr()->createQueryBuilder('s')
			->join('s.group', 'g')
			->orderBy('g.id', 'ASC')->addOrderBy('s.position', 'ASC');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('slidesGrid:gridSortableRow!');
		$grid->setSortableGroup('group');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:edit');
		$grid->addColumnText('group', 'default.group', 'group.title');
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('title', '');
		$groupsFilter = ['' => ''];
		foreach ($groups as $group)
			$groupsFilter[$group->getId()] = $group->title;
		$grid->getColumn('group')->setFilterSelect($groupsFilter, 's.group');

		// Actions
		$grid->addAction('edit', '', 'Default:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';


		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->slidesService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->slidesService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->slidesService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
	}
}
