<?php declare(strict_types = 1);

namespace Slides\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use Nette\Caching\Cache;
use Slides\FrontModule\Model\Dao;
use Slides\Model\Entities\Slide;

/**
 * Class Slides
 * @package Slides\FrontModule\Model
 *
 * @method Slide|object|null getReference($id)
 */
class Slides extends BaseEntityService
{
	protected $entityClass = Slide::class;

	const CACHE_NAMESPACE = 'slides';

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['slides'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];
	
	/** @var DefaultLang @inject */
	public $defaultLang;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param int $id Group id
	 *
	 * @return Dao\Slide[]
	 */
	public function getByGroup($id)
	{
		$locale = $this->defaultLang->locale;
		$key    = 'slidesByGroup_' . $locale . '_' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id, $locale) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'slidesByGroup/' . $id;

			$slides = $this->getEr()->createQueryBuilder('s')
				->where('s.isPublished = 1')
				->andWhere('s.lang = :lang')->setParameter('lang', $locale)
				->andWhere('s.group = :groupId')->setParameter('groupId', $id)
				->orderBy('s.position')
				->getQuery()->getResult(Query::HYDRATE_ARRAY);

			return $slides ? $this->fillDao($slides) : null;
		});
	}

	/**
	 * @param array $data
	 *
	 * @return Dao\Slide[]
	 */
	protected function fillDao($data)
	{
		$locale = $this->defaultLang->locale;
		$arr    = [];

		foreach ($data as $v)
			$arr[$v['id']] = (new Dao\Slide())
				->setId($v['id'])
				->setTitle($v['title'])
				->setLang($v['lang'])
				->setPosition($v['position'])
				->setTexts($v['texts']);

		return $arr;
	}
}
