<?php declare(strict_types = 1);

namespace Slides\AdminModule\Components;

use Blog\Model\UI;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Pages\Model\Paths;
use Slides\AdminModule\Model\Slides;
use Slides\Model\Entities\Slide;
use Slides\Model\SlideGroups;

class SlideForm extends TemplateReaderControl
{
	/** @var Slide */
	public $entity;

	/** @var Slides */
	protected $slidesService;

	/** @var SlideGroups */
	private $slideGroupsService;

	public function __construct(Slides $slides, SlideGroups $slideGroups)
	{
		$this->slidesService      = $slides;
		$this->slideGroupsService = $slideGroups;
	}

	public function render()
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render(__DIR__ . '/SlideForm.latte');
	}

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->slidesService->getTemplatesDir());
		$this->templateReader->setTranslateKey('templateSlide');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$groups = [];
		foreach ($this->slideGroupsService->getAll() as $g)
			$groups[$g->getId()] = $g->title;

		$form->addText('title', 'default.title')->setMaxLength(255)->setRequired();
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('group', 'default.group', $groups)->setRequired();
		$form->addLangsSelect('lang', 'default.language')->setItems();
		$form->addSelect('template', $this->t('default.templates'), $templates)->setPrompt($this->t('default.choose'))
			->setTranslator(null)->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		try {
			$group = $this->slideGroupsService->getReference($values->group);

			if ($this->entity) {
				$slide        = $this->entity;
				$flashMessage = 'slides.slideForm.edited';
			} else {
				$slide        = new Slide($values->title, $group, $values->template);
				$flashMessage = 'slides.slideForm.added';
			}

			$slide->title       = $values->title;
			$slide->isPublished = $values->isPublished;
			$slide->lang        = $values->lang;
			$slide->group       = $group;
			$slide->template    = $values->template;
			$slide->setTexts($form->getComponent('component')->getValues(true));

			$this->em->persist($slide)->flush();
			$form->addCustomData('slideId', $slide->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setSlide($id)
	{
		$this->entity = $this->em->getRepository(Slide::class)->find($id);

		if ($this->entity) {
			$s = $this->entity;
			$f = $this['form'];

			$f->setDefaults([
				'title'       => $s->title,
				'isPublished' => $s->isPublished,
				'lang'        => $s->lang,
			]);

			if ($s->group && array_key_exists($s->group->getId(), $f['group']->getItems()))
				$f['group']->setDefaultValue($s->group->getId());

			if ($s->template && array_key_exists($s->template, $f['template']->getItems())) {
				$f['template']->setDefaultValue($s->template);

				if ($this->httpRequest->getQuery('do') != 'slideForm-loadInputs') {
					$this->templateReader->loadTemplateComponents($f['component'], $this->httpRequest->getPost('template') ?: $s->template);
					$this->templateReader->setDefaults($f['component'], $s, $s->template);
				}
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
